package com.xebialabs.xlrelease.views.converters

import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind
import com.xebialabs.xlrelease.builder.ReleaseBuilder.newRelease
import com.xebialabs.xlrelease.domain.{BaseConfiguration, Release}
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.variable.VariableViewHelper.variableViewsToVariables
import com.xebialabs.xlrelease.views.ReleaseForm
import com.xebialabs.xlrelease.views.converters.DurationViewConverter.fromDurationView
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.jdk.CollectionConverters._

@Component
class ReleaseFormConverter @Autowired()(configurationRepository: ConfigurationRepository) {

  def toRelease(form: ReleaseForm): Release = {
    val release = newRelease()
      .withKind(form.getKind)
      .withTitle(form.getTitle)
      .withDescription(form.getDescription)
      .withOwner(form.getOwnerUsername)
      .withScheduledStartDate(form.getScheduledStartDate)
      .withAutoStart(form.getAutoStart)
      .withDueDate(form.getDueDate)
      .withPlannedDuration(fromDurationView(form.getPlannedDuration))
      .withVariables(variableViewsToVariables(form.getVariables))
      .withTags(form.getTags)
      .withFlagStatus(form.getFlag.getStatus)
      .withFlagComment(form.getFlag.getComment)
      .withCalendarPublished(form.isCalendarPublished)
      .withAbortOnFailure(form.isAbortOnFailure)
      .withArchiveRelease(form.isArchiveRelease)
      .withAllowPasswordsInAllFields(form.isAllowPasswordsInAllFields)
      .withDisableNotifications(form.isDisableNotifications)
      .withAllowConcurrentReleasesFromTrigger(form.isAllowConcurrentReleaseFromTrigger)
      .withScriptUsername(getScriptUsername(form).orNull)
      .withScriptUserPassword(form.getScriptUserPassword)
      .withVariableMapping(form.getVariableMapping)
      .withAuthor(form.getAuthor)
      .withCategories(new java.util.HashSet(form.getCategories))
      .withDefaultTargetFolderId(form.getDefaultTargetFolderId)
      .withAllowTargetFolderOverride(form.isAllowTargetFolderOverride)
      .build()
    fillProperties(release, form)
    release
  }

  private def getScriptUsername(form: ReleaseForm): Option[String] = {
    for {
      userView <- Option(form.getScriptUsername)
      name <- Option(userView.getUsername)
    } yield name
  }

  private def fillProperties(release: Release, form: ReleaseForm): Unit = {
    for (pd <- release.getType.getDescriptor.getPropertyDescriptors.asScala if form.hasProperty(pd.getName)) {
      pd.getKind match {
        case PropertyKind.CI =>
          val ciId = form.getProperty(pd.getName).asInstanceOf[String]
          val configuration = configurationRepository.read[BaseConfiguration](ciId)
          if (configuration != null) {
            checkArgument(configuration.getType.getDescriptor.isAssignableTo(pd.getReferencedType),
              s"CI of type ${configuration.getType.getName} is not assignable to property of type ${pd.getReferencedType.getName}")
            pd.set(release, configuration)
          }
        case _ => // don't care for now...
      }
    }
  }

}
