package com.xebialabs.xlrelease.api.internal

import com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank
import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.security.PermissionDeniedException
import com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN
import com.xebialabs.xlrelease.api.ApiService.{DEFAULT_RESULTS_PER_PAGE, PAGE, RESULTS_PER_PAGE}
import com.xebialabs.xlrelease.api.v1.forms.CreatePersonalAccessTokenForm
import com.xebialabs.xlrelease.domain.UserToken
import com.xebialabs.xlrelease.features.PersonalAccessTokenFeature
import com.xebialabs.xlrelease.principaldata.{PrincipalDataProvider, UserData}
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.service.{UserTokenService, Users}
import com.xebialabs.xlrelease.user.User
import grizzled.slf4j.Logging
import io.micrometer.core.annotation.Timed
import org.springframework.data.domain.{Page, PageRequest}
import org.springframework.stereotype.Controller

import jakarta.ws.rs._
import jakarta.ws.rs.core.MediaType
import scala.jdk.CollectionConverters._

@Path("/tokens")
@Consumes(Array(MediaType.APPLICATION_JSON))
@Produces(Array(MediaType.APPLICATION_JSON))
@Controller
class UserTokenResource(permissions: PermissionChecker,
                        tokenService: UserTokenService,
                        users: Users,
                        principalDataProvider: PrincipalDataProvider)
  extends Logging {

  @GET
  @Path("/users")
  @Timed
  def getUserTokens(@DefaultValue("0") @QueryParam(PAGE) page: Int,
                    @DefaultValue("100") @QueryParam(RESULTS_PER_PAGE) resultsPerPage: Int): Page[UserToken] = {
    getUserTokens(User.AUTHENTICATED_USER.getName, page, resultsPerPage)
  }

  @GET
  @Path("/users/{username}")
  def getTokensForUsername(@PathParam("username") username: String,
                           @DefaultValue("0") @QueryParam(PAGE) page: Int,
                           @DefaultValue("100") @QueryParam(RESULTS_PER_PAGE) resultsPerPage: Int): Page[UserToken] = {
    permissions.check(ADMIN)
    getUserTokens(username, page, resultsPerPage)
  }

  @POST
  @Path("/users/{username}")
  def generateUserToken(@PathParam("username") username: String,
                        userTokenForm: CreatePersonalAccessTokenForm): UserToken = {
    permissions.check(ADMIN)
    checkArgument(users.userExistsInRepository(username),
      "Personal access tokens generated by administrator are only supported for internal users")
    validateAndNormalizeUserTokenForm(username, userTokenForm)
    tokenService.createUserToken(username,
      userTokenForm.getTokenNote,
      userTokenForm.getExpiryDate,
      userTokenForm.getGlobalPermissions.asScala.toSet)
  }

  @POST
  @Path("/users")
  @Timed
  def generateUserToken(userTokenForm: CreatePersonalAccessTokenForm): UserToken = {
    val username: String = User.AUTHENTICATED_USER.getName
    val userData: UserData = principalDataProvider.getUserData(username)
    val personalAccessTokenSupported: Boolean = PersonalAccessTokenFeature.enabled || users.userExistsInRepository(username) || userData.isFound
    checkArgument(personalAccessTokenSupported,
      "Personal access tokens are only supported for internal users and external users authenticated via LDAP")
    validateAndNormalizeUserTokenForm(username, userTokenForm)
    tokenService.createUserToken(username,
      userTokenForm.getTokenNote,
      userTokenForm.getExpiryDate,
      userTokenForm.getGlobalPermissions.asScala.toSet)
  }

  @DELETE
  @Path("/users/{ciUid}")
  @Timed
  def deleteUserToken(@PathParam("ciUid") ciUid: Integer): Unit = {
    val userToken = tokenService.findByCiUid(ciUid)
    userToken match {
      case Some(token) =>
        if (token.username.equalsIgnoreCase(User.AUTHENTICATED_USER.getName) || permissions.isCurrentUserAdmin) {
          tokenService.deleteUserToken(token)
        } else {
          throw new PermissionDeniedException("You don't have the permissions required to delete this personal access token.")
        }
      case None => throw new IllegalArgumentException(s"Personal access token ciUid[$ciUid] is not valid")
    }
  }

  private def validateAndNormalizeUserTokenForm(username: String, userTokenForm: CreatePersonalAccessTokenForm): Unit = {
    checkArgument(isNotBlank(userTokenForm.getTokenNote), "Note can't be blank")
    checkArgument(userTokenForm.getExpiryDate == null || userTokenForm.getExpiryDate.getTime > System.currentTimeMillis(), "Expiry date is not valid")

    val tokenNote = userTokenForm.getTokenNote.trim
    userTokenForm.setTokenNote(tokenNote)

    checkArgument(tokenNote.length < 256, "Note must be 255 characters or less")
    checkArgument(!tokenService.userTokenExists(username, tokenNote), "Note has already been taken")
  }

  private def getUserTokens(username: String, page: Int, resultsPerPage: Int): Page[UserToken] = {
    checkArgument(resultsPerPage <= DEFAULT_RESULTS_PER_PAGE, "Number of results per page cannot be more than 100")
    val pageable = PageRequest.of(page, resultsPerPage)
    tokenService.find(username, pageable)
  }
}
