package com.xebialabs.xlrelease.repository;

import java.util.*;
import java.util.stream.Collectors;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.services.Repository;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.xlrelease.utils.CiHelper;
import com.xebialabs.xltype.serialization.CiReader;
import com.xebialabs.xltype.serialization.ConfigurationItemConverter;

import io.micrometer.core.annotation.Timed;

import static com.xebialabs.xlrelease.utils.CiHelper.getNestedCis;

public class CiCloneHelper {

    @Timed
    public static <R extends ConfigurationItem, T extends ConfigurationItem> R cloneAs(T ci, Type targetType) {
        Map<String, ConfigurationItem> cisWithRewrittenNullId = replaceNullIds(ci);

        InMemoryRepository externalReferences = new InMemoryRepository(CiHelper.getExternalReferences(ci));

        ConfigurationItemConverter ciConverter = new ConfigurationItemConverter();
        // TODO -> make sure target Type and ci type have common "type"
        CiReader ciReader = new CiMemoryReader(ci, targetType);
        @SuppressWarnings("unchecked")
        R clonedCi = (R) ciConverter.readCi(ciReader);

        ciConverter.resolveReferences(externalReferences);

        nullifyReplacedIds(ci, cisWithRewrittenNullId);
        nullifyReplacedIds(clonedCi, cisWithRewrittenNullId);

        return clonedCi;
    }

    @Timed
    public static <T extends ConfigurationItem> T cloneCi(T ci) {

        Map<String, ConfigurationItem> cisWithRewrittenNullId = replaceNullIds(ci);

        InMemoryRepository externalReferences = new InMemoryRepository(CiHelper.getExternalReferences(ci));

        ConfigurationItemConverter ciConverter = new ConfigurationItemConverter();
        CiReader ciReader = new CiMemoryReader(ci);
        @SuppressWarnings("unchecked")
        T clonedCi = (T) ciConverter.readCi(ciReader);

        ciConverter.resolveReferences(externalReferences);

        nullifyReplacedIds(ci, cisWithRewrittenNullId);
        nullifyReplacedIds(clonedCi, cisWithRewrittenNullId);

        return clonedCi;
    }

    @Timed
    public static <T extends ConfigurationItem> List<T> cloneCis(List<T> ciList) {
        return ciList.stream()
                .map(CiCloneHelper::cloneCi).collect(Collectors.toList());
    }

    private static Map<String, ConfigurationItem> replaceNullIds(ConfigurationItem parentCi) {
        Map<String, ConfigurationItem> replaced = new HashMap<>();
        getNestedCis(parentCi).stream()
                .filter(ci -> ci.getId() == null)
                .forEach(ci -> {
                    String temporaryId = UUID.randomUUID().toString();
                    ci.setId(temporaryId);
                    replaced.put(temporaryId, ci);
                });
        return replaced;
    }

    private static void nullifyReplacedIds(ConfigurationItem parentCi, Map<String, ConfigurationItem> replacedMapping) {
        getNestedCis(parentCi).forEach(ci -> {
            if (replacedMapping.containsKey(ci.getId())) {
                ci.setId(null);
            }
        });
    }

    static class InMemoryRepository implements Repository {

        private final Map<String, ConfigurationItem> cis;

        InMemoryRepository(Collection<ConfigurationItem> inputCis) {
            this.cis = new HashMap<>(inputCis.size());
            inputCis.forEach(ci -> this.cis.put(ci.getId(), ci));
        }

        @Override
        public boolean exists(final String id) {
            return cis.containsKey(id);
        }

        @Override
        @SuppressWarnings("unchecked")
        public <T extends ConfigurationItem> T read(final String id) {
            return (T) cis.get(id);
        }

        @Override
        public <T extends ConfigurationItem> List<T> read(final List<String> ids, final Integer depth) {
            return ids.stream().map(this::<T>read).filter(Objects::nonNull).collect(Collectors.toList());
        }

        @Override
        public <T extends ConfigurationItem> void create(final T[] entity) {
            throw new RuntimeException("Not supported in InMemoryRepository");
        }

        @Override
        public <T extends ConfigurationItem> void update(final T[] entity) {
            throw new RuntimeException("Not supported in InMemoryRepository");
        }

        @Override
        public <T extends ConfigurationItem> void createOrUpdate(final T[] entity) {
            throw new RuntimeException("Not supported in InMemoryRepository");
        }

        @Override
        public void delete(final String... id) {
            throw new RuntimeException("Not supported in InMemoryRepository");
        }

        @Override
        public void move(final String id, final String newId) {
            throw new RuntimeException("Not supported in InMemoryRepository");
        }

        @Override
        public void rename(final String id, final String newName) {
            throw new RuntimeException("Not supported in InMemoryRepository");
        }

        @Override
        public <T extends ConfigurationItem> List<T> search(final Type type) {
            throw new RuntimeException("Not supported in InMemoryRepository");
        }

        @Override
        public <T extends ConfigurationItem> List<T> search(final Type type, final String parent) {
            throw new RuntimeException("Not supported in InMemoryRepository");
        }
    }
}
