package com.xebialabs.xlrelease.service;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.SmartLifecycle;
import org.springframework.context.annotation.DependsOn;
import org.springframework.stereotype.Component;

import com.xebialabs.deployit.server.api.upgrade.Upgrade;
import com.xebialabs.deployit.server.api.upgrade.Version;
import com.xebialabs.deployit.upgrade.RepositoryInitializer;
import com.xebialabs.deployit.upgrade.UpgradeRejectedException;
import com.xebialabs.deployit.upgrade.Upgrader;
import com.xebialabs.xlplatform.spring.JythonBindingsHolder;
import com.xebialabs.xlplatform.upgrade.UpgraderHelper;
import com.xebialabs.xlrelease.api.ApiService;
import com.xebialabs.xlrelease.api.XLReleaseServiceHolder;
import com.xebialabs.xlrelease.config.TypedWebhookFeatureToggleCondition;
import com.xebialabs.xlrelease.config.XlrConfig;
import com.xebialabs.xlrelease.db.XLReleaseDbInitializer;
import com.xebialabs.xlrelease.events.EventListenerRegister;
import com.xebialabs.xlrelease.script.bindings.SecurityApiBinding;
import com.xebialabs.xlrelease.upgrade.Components;
import com.xebialabs.xlrelease.upgrade.liquibase.BeforeLiquibaseUpgrade;

@Component
@DependsOn("permissionCheckerHolder") // upgraders need permissionCheckerHolder to be initialized
public class ServiceStarter implements SmartLifecycle {
    public static final int LIFECYCLE_PHASE = -1;
    private boolean running = false;
    private final Upgrader upgrader;
    private final RepositoryInitializer repositoryInitializer;
    private final XlrServiceManager serviceManager;
    private final XlrConfig xlrConfig;
    private final MissingTypesChecker missingTypesChecker;
    private XLReleaseDbInitializer xlrMigrationsDbInitializer;
    private XLReleaseDbInitializer xlrRepositoryDbInitializer;
    private XLReleaseDbInitializer reportingDbInitializer;
    private EventListenerRegister eventListenerRegister;
    private BeforeLiquibaseUpgrade beforeLiquibaseUpgrade;
    private List<? extends ApiService> apiServices;
    private SecurityApiBinding securityApiBinding;
    private DatacenterService datacenterService;

    @Autowired
    public ServiceStarter(Upgrader upgrader,
                          RepositoryInitializer repositoryInitializer,
                          XlrServiceManager serviceManager,
                          XlrConfig xlrConfig,
                          MissingTypesChecker missingTypesChecker,
                          BeforeLiquibaseUpgrade beforeLiquibaseUpgrade,
                          @Qualifier("xlrMigrationsDbInitializer") XLReleaseDbInitializer xlrMigrationsDbInitializer,
                          @Qualifier("xlrRepositoryDbInitializer") XLReleaseDbInitializer xlrRepositoryDbInitializer,
                          @Qualifier("reportingDbInitializer") XLReleaseDbInitializer reportingDbInitializer,
                          EventListenerRegister eventListenerRegister,
                          SecurityApiBinding securityApiBinding,
                          DatacenterService datacenterService
    ) {
        this.upgrader = upgrader;
        this.repositoryInitializer = repositoryInitializer;
        this.serviceManager = serviceManager;
        this.xlrConfig = xlrConfig;
        this.missingTypesChecker = missingTypesChecker;
        this.beforeLiquibaseUpgrade = beforeLiquibaseUpgrade;
        this.xlrMigrationsDbInitializer = xlrMigrationsDbInitializer;
        this.xlrRepositoryDbInitializer = xlrRepositoryDbInitializer;
        this.reportingDbInitializer = reportingDbInitializer;
        this.eventListenerRegister = eventListenerRegister;
        this.securityApiBinding = securityApiBinding;
        this.datacenterService = datacenterService;
    }

    @Autowired
    public void setApiServices(List<? extends ApiService> apiServices) {
        this.apiServices = apiServices;
    }

    public void startServices() {
        if (!running) {
            boolean shutdown = false;
            Version xlrDataVersion;

            XLReleaseServiceHolder.init(apiServices);
            initEndpointsJythonBindings(apiServices);
            logScriptSecurityOptions();

            try {
                xlrDataVersion = upgrader.getComponentVersion(Components.XL_RELEASE_COMPONENT);
            } catch (RuntimeException e) {
                xlrDataVersion = null;
            }

            if (xlrDataVersion != null) {
                List<Upgrade> allUpgrades = upgrader.findUpgrades().get(Components.XL_RELEASE_COMPONENT);
                List<Upgrade> upgrades = UpgraderHelper.filterApplicable(allUpgrades, xlrDataVersion);
                allUpgrades.clear();

                if (!upgrades.isEmpty()) {
                    if (xlrConfig.isVerifyUpgraderMode()) {
                        throw new RuntimeException("Upgrade verification failed, run upgrader in apply mode.");
                    }
                    try {
                        upgrader.askForUpgrade();
                    } catch (UpgradeRejectedException e) {
                        shutdown = true;
                    }
                }
            }

            if (!shutdown) {
                beforeLiquibaseUpgrade.init();

                logger.info("Initializing databases.");

                xlrMigrationsDbInitializer.run();
                xlrRepositoryDbInitializer.run();
                reportingDbInitializer.run();

                repositoryInitializer.initializeComponents();

                runUpgradeFiles();
                missingTypesChecker.cleanup();
                if (new TypedWebhookFeatureToggleCondition().matches()) {
                    missingTypesChecker.disableMissingWebhookEventTypes();
                }
                eventListenerRegister.registerEventListeners();
                datacenterService.registerDatacenter();
                datacenterService.monitorDatacenterTargetState();
            }
        } else {
            logger.error("ServiceStarter is already running. Please check configuration and report a bug.");
        }
    }

    private void runUpgradeFiles() {
        boolean isVerifyMode = xlrConfig.isVerifyUpgraderMode();
        try {
            upgrader.autoUpgrade(isVerifyMode);
        } catch (UpgradeRejectedException exception) {
            if (isVerifyMode) {
                logger.error("Upgrade files verification failed: ", exception);
                throw new RuntimeException("Upgrade verification failed, run upgrader in apply mode.");
            } else {
                throw exception;
            }
        }
    }

    private void initEndpointsJythonBindings(List<? extends ApiService> apiServices) {
        Map<String, Object> bindings = new HashMap<>();
        bindings.put(SecurityApiBinding.NAME, securityApiBinding);
        apiServices.forEach(apiService -> bindings.put(apiService.serviceName(), apiService));

        JythonBindingsHolder.setBindings(Collections.unmodifiableMap(bindings));
    }

    private void logScriptSecurityOptions() {
        boolean hasSecurityManager = System.getSecurityManager() != null;
        boolean hasScriptSandbox = hasSecurityManager && xlrConfig.isScriptSandboxEnabled();
        logger.info("Configuring scripting engine - Java security is {}, script sandbox is {}.",
                hasSecurityManager ? "configured" : "not configured",
                hasScriptSandbox ? "enabled" : "disabled");
    }

    private static final Logger logger = LoggerFactory.getLogger(ServiceStarter.class);

    @Override
    public void start() {
        try {
            startServices();
            running = true;
        } catch (Exception e) {
            logger.error("Failed to start services", e);
            throw e;
        }
    }

    @Override
    public void stop() {
        datacenterService.cancelStateCheck();
        serviceManager.stop();
        running = false;
    }

    @Override
    public boolean isRunning() {
        return running;
    }

    @Override
    public int getPhase() {
        return LIFECYCLE_PHASE; // make this a priority
    }
}
