package com.xebialabs.xlrelease.upgrade.db

import com.xebialabs.deployit.server.api.upgrade.{Upgrade, Version}
import com.xebialabs.xlrelease.api.v1.forms.ReleasesFilters
import com.xebialabs.xlrelease.db.ArchivedReleases._
import com.xebialabs.xlrelease.db.ArchivedReleasesSearch
import com.xebialabs.xlrelease.export.TemplateJsonHelper.getTasksFromRelease
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import grizzled.slf4j.Logging
import org.apache.commons.lang.StringUtils.isNotBlank
import org.codehaus.jettison.json.{JSONException, JSONObject}
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{BatchPreparedStatementSetter, JdbcTemplate}
import org.springframework.stereotype.Component
import org.springframework.transaction.TransactionStatus
import org.springframework.transaction.support.TransactionTemplate

import java.sql.PreparedStatement
import scala.annotation.tailrec
import scala.jdk.CollectionConverters._

@Component
class XLRelease750ReportingSchemaUpgrade @Autowired()(@Qualifier("reportingJdbcTemplate") jdbcTemplate: JdbcTemplate,
                                                      @Qualifier("reportTransactionTemplate") transactionTemplate: TransactionTemplate,
                                                      archivedReleasesSearch: ArchivedReleasesSearch
                                                     ) extends Upgrade with Logging {

  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "7.5.0#3")

  private val BATCH_SIZE = 100

  override def doUpgrade(): Boolean = {
    searchAndProcessReleases()
    true
  }

  @tailrec
  private def searchAndProcessReleases(page: Int = 0, processedCount: Int = 0): Int = {
    val infoEvery = 1000
    if (page * BATCH_SIZE % infoEvery == 0) {
      logger.info(s"Processing archived releases ${page * BATCH_SIZE} - ${page * BATCH_SIZE + infoEvery}")
    }
    searchReleases(page, BATCH_SIZE) match {
      case Nil =>
        logger.info(s"Finished processing $processedCount archived releases")
        processedCount
      case releasesBatch =>
        processReleases(releasesBatch)
        searchAndProcessReleases(page + 1, processedCount + releasesBatch.size)
    }
  }

  private def searchReleases(page: Long, pageSize: Long): List[String] = {
    val filters = new ReleasesFilters()
    filters.setInactive(true)
    archivedReleasesSearch.searchReleases(filters, Some(pageSize), Some(page * pageSize))
  }

  private def processReleases(releases: Seq[String]): Unit = {
    val parsedReleases = releases.flatMap { releaseContent =>
      try {
        Option(new JSONObject(releaseContent))
      } catch {
        case e: JSONException =>
          logger.error(s"Could not process an archived release with JSON content:\n$releaseContent", e)
          None
      }
    }
    transactionTemplate.execute((_: TransactionStatus) => {
      updateOriginalTemplateId(parsedReleases
        .filter(json => isNotBlank(json.optString("id")) && isNotBlank(json.optString("originTemplateId"))).toList)
      updateTaskTypes(parsedReleases.filter(json => isNotBlank(json.optString("id"))).toList)
    })
  }

  private def updateOriginalTemplateId(releasesJson: List[JSONObject]): Unit = {

    jdbcTemplate.batchUpdate(
      s"""UPDATE $REPORT_RELEASES_TABLE_NAME
         |SET $REPORT_RELEASES_ORIGIN_TEMPLATE_ID = ?
         |WHERE $REPORT_RELEASES_ID_COLUMN = ?""".stripMargin,
      new BatchPreparedStatementSetter {

        override def getBatchSize: Int = releasesJson.size

        override def setValues(ps: PreparedStatement, i: Int): Unit = {
          val json = releasesJson(i)
          val releaseId = json.optString("id")
          val templateId = json.optString("originTemplateId")

          ps.setString(1, originalTemplateIdToColumnValue(templateId))
          ps.setString(2, shortenId(releaseId))
        }

      }
    )
  }


  private def updateTaskTypes(releasesJson: List[JSONObject]): Unit = {

    releasesJson.foreach { releaseJson =>
      val releaseId = releaseJson.optString("id")
      val tasks = getTasksFromRelease(releaseJson).asScala.filter(filterByTaskTypes).toList
      jdbcTemplate.batchUpdate(
        // Extra constraint on releaseId here is added for better performance as taskId is not indexed
        s"""UPDATE $REPORT_TASKS_TABLE_NAME
           |SET $REPORT_TASKS_TASK_TYPE_COLUMN = ?
           |WHERE
           |  $REPORT_TASKS_RELEASEID_COLUMN = ?
           |  AND $REPORT_TASKS_ID_COLUMN = ?""".stripMargin,
        new BatchPreparedStatementSetter {

          override def getBatchSize: Int = tasks.size

          override def setValues(ps: PreparedStatement, i: Int): Unit = {

            val taskJson = tasks(i)
            val taskId = taskJson.optString("id")
            val taskType = if (taskJson.optJSONObject("pythonScript") != null) {
              taskJson.getJSONObject("pythonScript").optString("type")
            } else {
              taskJson.optString("type")
            }
            ps.setString(1, taskType)
            ps.setString(2, shortenId(releaseId))
            ps.setString(3, taskId) // not shortening here because before 7.5.0 task IDs were not shortened
          }
        }
      )
    }
  }

  private def filterByTaskTypes(taskJson: JSONObject): Boolean = {
    val taskId = taskJson.optString("id")
    val taskType = if (taskJson.optJSONObject("pythonScript") != null) {
      taskJson.getJSONObject("pythonScript").optString("type")
    } else {
      taskJson.optString("type")
    }
    isNotBlank(taskType) && isNotBlank(taskId)
  }
}
