package com.xebialabs.xlrelease.service

import com.xebialabs.xlrelease.config.ArchivingSettingsManager
import grizzled.slf4j.Logging
import org.joda.time.LocalDateTime
import org.springframework.stereotype.Service

@Service
class ArchivingScheduleService(archivingService: ArchivingService,
                               archivingConfig: ArchivingSettingsManager,
                               releaseSearchService: ReleaseSearchService)
  extends BaseBackgroundJobService[String] {

  val actions = List(
    new ArchiveAction(archivingService),
    new DeleteExpiredAction(releaseSearchService, archivingService)
  )

  override def jobDisplayName: String = "archiving"

  override def isJobEnabled(): Boolean = {
    archivingConfig.getEnabled
  }

  override def getCronSchedule(): String = {
    archivingConfig.getArchivingJobCronSchedule
  }

  override def getPeriodInHours(): Int = {
    archivingConfig.getReleaseAgeToDeleteFromJcr
  }

  override def getJobActions(): List[BackgroundJobAction[String]] = {
    actions
  }

  override def getMaxSecondsForJob(): Int = {
    archivingConfig.getMaxSecondsPerRun
  }

  override def getConfiguredPageSize(): Int = {
    archivingConfig.getSearchPageSize
  }

  override def getSleepSecondsBetweenItems(): Int = {
    archivingConfig.getSleepSecondsBetweenReleases
  }

  def processExpiredReleases(): Unit = {
    executeJob()
  }

  def processExpiredReleases(periodInHours: Int): Unit = {
    executeJob(periodInHours)
  }

}

// Archive completed/aborted releases and workflows
class ArchiveAction(archivingService: ArchivingService) extends BackgroundJobAction[String] with Logging {
  override def queryIdentifiers(jobStartTime: LocalDateTime, jobQueryPeriodInHours: Int, pageSize: Int): List[String] = {
    val expirationDate = jobStartTime.minusHours(jobQueryPeriodInHours).toDate()
    archivingService.findArchivableReleaseIds(expirationDate, pageSize).toList
  }

  override def processIdentifier(identifier: String): Unit = {
    logger.info(s"Archiving release [$identifier]")
    archivingService.archiveRelease(identifier)
  }
}

// Delete release/workflows that have archiving disabled and are older than the archiving age
class DeleteExpiredAction(releaseSearchService: ReleaseSearchService,
                          archivingService: ArchivingService) extends BackgroundJobAction[String] with Logging {
  override def queryIdentifiers(jobStartTime: LocalDateTime, jobQueryPeriodInHours: Int, pageSize: Int): List[String] = {
    // 15 minutes grace for pre-archival
    val expirationDate = jobStartTime.minusHours(jobQueryPeriodInHours).minusMinutes(15).toDate()
    releaseSearchService.findNonArchivedExpiredReleaseIds(expirationDate, pageSize).toList
  }

  override def processIdentifier(identifier: String): Unit = {
    logger.info(s"Removing expired release [$identifier] that won't be archived")
    // Looks weird but this handles deleting releases if they aren't supposed to be archived
    archivingService.archiveRelease(identifier)
  }
}
