package com.xebialabs.deployit.plumbing;

import java.util.Date;
import jakarta.ws.rs.NotAllowedException;
import jakarta.ws.rs.NotFoundException;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.ext.ExceptionMapper;
import jakarta.ws.rs.ext.Provider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import com.fasterxml.jackson.core.exc.StreamConstraintsException;

import com.xebialabs.deployit.ServerConfiguration;
import com.xebialabs.deployit.rest.mappers.UnhandledExceptionMapper;
import com.xebialabs.xlrelease.rest.mappers.StreamConstraintsExceptionMapper;

import static jakarta.ws.rs.core.Response.Status.*;
import static jakarta.ws.rs.core.Response.fromResponse;

@Provider
@Component
public class UnhandledExceptionMapperProvider extends UnhandledExceptionMapper implements ExceptionMapper<RuntimeException> {

    private final StreamConstraintsExceptionMapper constraintsExceptionMapper;

    @Autowired
    public UnhandledExceptionMapperProvider(ServerConfiguration config, StreamConstraintsExceptionMapper constraintsExceptionMapper) {
        super(config);
        this.constraintsExceptionMapper = constraintsExceptionMapper;
    }

    @Override
    protected Response handleException(RuntimeException runtimeException) {
        Response response = super.handleException(runtimeException);
        response.getMetadata().add("Date", new Date());

        if (runtimeException instanceof NotFoundException) {
            return fromResponse(response).status(NOT_FOUND).build();
        } else if (runtimeException instanceof IllegalArgumentException) {
            return fromResponse(response).status(BAD_REQUEST).entity(runtimeException.getMessage()).build();
        } else if (runtimeException instanceof NotAllowedException) {
            return fromResponse(response).status(METHOD_NOT_ALLOWED).entity(runtimeException.getMessage()).build();
        } else if (runtimeException.getCause() instanceof StreamConstraintsException) {
            return constraintsExceptionMapper.toResponse(new StreamConstraintsException(runtimeException.getCause().getMessage()));
        }

        return response;
    }
}
