package com.xebialabs.xlrelease.service;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.xebialabs.deployit.engine.api.security.Role;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xlrelease.api.internal.DecoratorsCache;
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.Team;
import com.xebialabs.xlrelease.domain.events.*;
import com.xebialabs.xlrelease.domain.folder.Folder;
import com.xebialabs.xlrelease.events.XLReleaseEventBus;
import com.xebialabs.xlrelease.initialize.TutorialsFolderInitializer;
import com.xebialabs.xlrelease.repository.*;
import com.xebialabs.xlrelease.security.SecuredCi;
import com.xebialabs.xlrelease.utils.Diff;
import com.xebialabs.xlrelease.views.TeamView;
import com.xebialabs.xlrelease.views.converters.TeamMemberViewConverter;

import io.micrometer.core.annotation.Timed;
import scala.Tuple2;
import scala.jdk.javaapi.StreamConverters;

import static com.google.common.base.Preconditions.checkArgument;
import static com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.CONFIGURATION;
import static com.xebialabs.xlrelease.api.internal.DecoratorsCache.NO_CACHE;
import static com.xebialabs.xlrelease.repository.CiCloneHelper.cloneCi;
import static com.xebialabs.xlrelease.repository.Ids.getName;
import static com.xebialabs.xlrelease.repository.Ids.getParentId;
import static com.xebialabs.xlrelease.repository.Ids.isInFolder;
import static com.xebialabs.xlrelease.repository.Ids.isNullId;
import static com.xebialabs.xlrelease.repository.Ids.releaseIdFrom;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toSet;
import static scala.jdk.javaapi.CollectionConverters.asJava;

@Service
@IsTransactional
public class TeamService implements TeamOperationsService {
    public static final String GLOBAL_ROLES_ROOT = CONFIGURATION.getRootNodeName() + "/security/global";
    public static final String CACHE_KEY_SECURED_CI_FOR_CONTAINER_ID = "securedCiForContainerId:";
    public static final String CACHE_KEY_TEAMS_FOR_SECURED_CI = "teamsForSecuredCi:";

    private final TeamRepository teamRepository;
    private final ReleaseRepository releaseRepository;
    private final SecuredCis securedCis;
    private final XLReleaseEventBus eventBus;
    private final CiIdService ciIdService;
    private final SecurityRepository securityRepository;
    public final ArchivingService archivingService;

    @Autowired
    public TeamService(final TeamRepository teamRepository,
                       final ReleaseRepository releaseRepository,
                       final ArchivingService archivingService,
                       final SecuredCis securedCis,
                       final XLReleaseEventBus eventBus,
                       final CiIdService ciIdService,
                       final SecurityRepository securityRepository
                       ) {
        this.teamRepository = teamRepository;
        this.releaseRepository = releaseRepository;
        this.archivingService = archivingService;
        this.securedCis = securedCis;
        this.eventBus = eventBus;
        this.ciIdService = ciIdService;
        this.securityRepository = securityRepository;
    }

    @Timed
    public Team addTeam(Release release, Team team) {
        checkArgument(release.isUpdatable(), "Can't add team to release '%s' because it is %s", release.getTitle(), release.getStatus());
        checkArgument(!isInFolder(release.getId()), "Can't add team to release '%s' because it is inside a folder", release.getTitle());

        generateIdIfNecessary(release.getId(), team);
        Team addedTeam = teamRepository.create(release.getId(), team);
        release.addTeam(addedTeam);

        eventBus.publish(TeamCreatedEvent.apply(release, addedTeam));
        eventBus.publish(TeamsUpdatedEvent.apply(release.getId()));

        return addedTeam;
    }

    @Timed
    public Team addTeam(String containerId, Team team) {
        generateIdIfNecessary(containerId, team);
        Team addedTeam = teamRepository.create(containerId, team);
        eventBus.publish(TeamCreatedEvent.apply(containerId, addedTeam));
        eventBus.publish(TeamsUpdatedEvent.apply(containerId));
        return addedTeam;
    }

    @Timed
    public Team updateTeam(String teamId, Team newTeam) {
        String releaseId = releaseIdFrom(teamId);
        Release release = releaseRepository.findById(releaseId);
        checkArgument(release.isUpdatable(), "Can't update team on release '%s' because it is %s", release.getTitle(), release.getStatus());
        checkArgument(!isInFolder(releaseId), "Can't update team on release '%s' because it is inside a folder", release.getTitle());

        decorateWithStoredTeams(release);
        Team team = release.getTeamWithId(teamId);

        if (team.isSystemTeam()) {
            checkArgument(newTeam.getTeamName().equals(team.getTeamName()), "Cannot rename a system release");
        }

        release.updateTeam(newTeam);

        saveTeamsToPlatform(release, false);
        eventBus.publish(TeamUpdatedEvent.apply(release, team, newTeam));
        eventBus.publish(TeamsUpdatedEvent.apply(release.getId()));

        return team;
    }

    @Timed
    public List<Team> updateTeams(String releaseId, final List<Team> updatedTeams) {
        Release release = releaseRepository.findById(releaseId);
        checkArgument(release.isUpdatable(), "Can't update teams on release '%s' because it is %s", release.getTitle(), release.getStatus());
        checkArgument(!isInFolder(release.getId()), "Can't update teams on release '%s' because it is inside a folder", release.getTitle());

        decorateWithStoredTeams(release);
        List<Team> currentTeams = release.getTeams();
        Map<String, Set<String>> currentPermissions = getTeamsPermissions(currentTeams);

        Set<String> updatedTeamIds = updatedTeams.stream().map(Team::getId).collect(toSet());
        Set<String> currentTeamIds = currentTeams.stream().map(Team::getId).collect(toSet());

        Set<Team> toDelete = currentTeams.stream().filter(team -> !updatedTeamIds.contains(team.getId())).collect(toSet());
        Set<Team> toUpdate = updatedTeams.stream().filter(team -> currentTeamIds.contains(team.getId())).collect(toSet());
        Set<Team> toCreate = updatedTeams.stream().filter(team -> !currentTeamIds.contains(team.getId())).collect(toSet());

        List<XLReleaseEvent> teamEvents = new ArrayList<>();

        toDelete.forEach(team -> {
            release.deleteTeam(team.getId());
            teamEvents.add(TeamDeletedEvent.apply(release, team));
        });
        toUpdate.forEach(team -> {
            Team currentTeam = release.getTeamWithId(team.getId());
            Team original = cloneCi(currentTeam);
            release.updateTeam(team);
            if (!original.equals(team)) {
                teamEvents.add(TeamUpdatedEvent.apply(release, original, team));
            }
        });
        toCreate.forEach(team -> {
            release.addTeam(team);
            teamEvents.add(TeamCreatedEvent.apply(release, team));
        });

        Map<String, Set<String>> newPermissions = getTeamsPermissions(release.getTeams());
        if (!newPermissions.equals(currentPermissions)) {
            teamEvents.add(new PermissionsUpdatedEvent(release.getTeams()));
        }

        saveTeamsToPlatform(release, false);
        teamEvents.forEach(eventBus::publish);
        if (!teamEvents.isEmpty()) {
            eventBus.publish(TeamsUpdatedEvent.apply(releaseId));
        }

        return release.getTeams();
    }

    @Timed
    public void deleteTeam(Release release, String teamId) {
        checkArgument(release.isUpdatable(), "Can't delete team from release '%s' because it is %s", release.getTitle(), release.getStatus());
        checkArgument(!isInFolder(release.getId()), "Can't delete team from release '%s' because it is inside a folder", release.getTitle());

        Team team = release.getTeamWithId(teamId);

        release.deleteTeam(teamId);
        saveTeamsToPlatform(release, false);

        eventBus.publish(TeamDeletedEvent.apply(release, team));
        eventBus.publish(TeamsUpdatedEvent.apply(release.getId()));
    }

    @Timed
    public void deleteTeam(String containerId, String teamId) {
        List<Team> teams = getEffectiveTeams(containerId);
        Team toDelete = teams.stream().filter(team -> team.getId().equals(teamId)).findFirst().orElseThrow(
                () -> new IllegalStateException("Container " + containerId + " doesn't contain team with id " + teamId)
        );
        teamRepository.delete(teamId);
        eventBus.publish(TeamDeletedEvent.apply(containerId, toDelete));
        eventBus.publish(TeamsUpdatedEvent.apply(containerId));

    }

    public List<Team> saveTeamsToPlatform(Release release) {
        return saveTeamsToPlatform(release, true);
    }

    private List<Team> saveTeamsToPlatform(Release release, boolean publishEvent) {
        return saveTeamsToPlatform(release.getId(), release.getTeams(), publishEvent);
    }

    public List<Team> saveTeamsToPlatform(String containerId, List<Team> teams) {
        return saveTeamsToPlatform(containerId, teams, true);
    }

    private List<Team> saveTeamsToPlatform(String containerId, List<Team> teams, boolean publishEvents) {
        Tuple2<List<Team>, List<XLReleaseEvent>> result = saveTeamsToPlatformWithoutPublishing(containerId, teams, publishEvents);

        if (publishEvents) {
            result._2.forEach(eventBus::publish);
        }

        return result._1;
    }

    public Tuple2<List<Team>, List<XLReleaseEvent>> saveTeamsToPlatformWithoutPublishing(String containerId, List<Team> teams, boolean buildEvents) {
        Optional<Team> invalidTeam = teams.stream().filter(team -> !isNullId(team.getId()) && !containerId.equals(getParentId(team.getId()))).findFirst();
        checkArgument(invalidTeam.isEmpty(), "Cannot save team '%s' as it does not belong to '%s'", invalidTeam, containerId);

        teams.forEach(team -> {
            if (team.getTeamName() != null) team.setTeamName(team.getTeamName().trim());
            generateIdIfNecessary(containerId, team);
        });

        List<Team> originalTeams = buildEvents ? getStoredTeams(containerId) : null;//Fetch original teams (if needed) before updating platform
        List<Team> newTeams = teamRepository.saveTeamsToPlatform(containerId, teams);

        List<XLReleaseEvent> events = new ArrayList<>();
        if (buildEvents) {
            Diff<String, Team> differences = Diff.applyWithKeyMapping(originalTeams, newTeams, Team::getId);

            asJava(differences.deletedValues())
                    .forEach(t -> events.add(TeamDeletedEvent.apply(containerId, t)));
            asJava(differences.newValues())
                    .forEach(t -> events.add(TeamCreatedEvent.apply(containerId, t)));
            StreamConverters.asJavaSeqStream(differences.updatedPairs())
                    .filter(this::areTeamsDifferent)
                    .forEach(teamTuple2 -> events.add(TeamUpdatedEvent.apply(containerId, teamTuple2._1, teamTuple2._2)));

            events.add(TeamsUpdatedEvent.apply(containerId));
        }
        return new Tuple2<>(newTeams, events);
    }

    private boolean areTeamsDifferent(Tuple2<Team, Team> updatedPair) {
        Team before = updatedPair._1;
        Team after = updatedPair._2;
        return !before.equals(after);
    }

    public void deleteTeamsFromPlatform(String containerId) {
        List<Team> teamsToDelete = this.getEffectiveTeams(containerId);
        teamRepository.deleteTeamsFromPlatform(containerId);
        teamsToDelete.forEach(
                team -> eventBus.publish(TeamDeletedEvent.apply(containerId, team))
        );
        eventBus.publish(TeamsUpdatedEvent.apply(containerId));
    }

    // effective teams

    public Optional<Team> findTeamByName(String containerId, String teamName) {
        return findTeamsByNames(containerId, singletonList(teamName)).findFirst();
    }

    public Stream<Team> findTeamsByNames(String containerId, Collection<String> teamNames, DecoratorsCache cache) {
        return getEffectiveTeams(containerId, cache).stream().filter(team -> teamNames.contains(team.getTeamName()));
    }

    public Stream<Team> findTeamsByNames(String containerId, Collection<String> teamNames) {
        return getEffectiveTeams(containerId).stream().filter(team -> teamNames.contains(team.getTeamName()));
    }

    public void decorateWithEffectiveTeams(Release release) {
        decorateWithEffectiveTeams(release, NO_CACHE());
        release.setTeams(getEffectiveTeams(release));
    }

    public void decorateWithEffectiveTeams(Release release, DecoratorsCache cache) {
        release.setTeams(getEffectiveTeams(release, cache));
    }

    public List<Team> getEffectiveTeams(Release release) {
        return getEffectiveTeams(release, NO_CACHE());
    }

    public List<Team> getEffectiveTeams(Release release, DecoratorsCache cache) {
        if (release.isArchived()) {
            return release.getTeams();
        } else {
            String containerId = release.getId();
            return getEffectiveTeamsForContainerId(containerId, cache);
        }
    }

    private List<Team> getEffectiveTeamsForContainerId(final String containerId, final DecoratorsCache cache) {
        SecuredCi securedCi = cache.
                <SecuredCi>getDecoratorCache(CACHE_KEY_SECURED_CI_FOR_CONTAINER_ID + (containerId == null ? "" : containerId))
                .computeIfAbsent(() -> securedCis.getEffectiveSecuredCi(containerId));
        return cache.<List<Team>>getDecoratorCache(CACHE_KEY_TEAMS_FOR_SECURED_CI + (securedCi == null ? "" : securedCi.getId() + ":" + securedCi.getSecurityUid())).computeIfAbsent(() -> teamRepository.getTeams(securedCi));
    }

    public List<Team> getEffectiveTeams(Folder folder) {
        return teamRepository.getTeams(folder.get$internalId() != null && folder.get$internalId().equals(folder.get$securedCi()) ? new SecuredCi(folder.getId(), folder.get$internalId()) : securedCis.getEffectiveSecuredCi(folder.getId()));
    }

    public List<Team> getEffectiveTeams(String containerId) {
        return getEffectiveTeams(containerId, NO_CACHE());
    }

    public List<Team> getEffectiveTeams(String containerId, DecoratorsCache cache) {
        if (isArchivedRelease(containerId)) {
            return archivingService.getRelease(containerId).getTeams();
        } else {
            return getEffectiveTeamsForContainerId(containerId, cache);
        }
    }

    public List<TeamView> getEffectiveTeamViews(String containerId, TeamMemberViewConverter teamMemberViewConverter) {
        if (isArchivedRelease(containerId)) {
            List<Team> teams = archivingService.getRelease(containerId).getTeams();
            return teams.stream()
                    .map(team -> new TeamView(team, teamMemberViewConverter))
                    .sorted(Comparator.comparing(TeamView::getTeamName))
                    .toList();
        } else {
            return getTeamViews(containerId);
        }
    }

    public void decorateWithStoredTeams(Release release) {
        release.setTeams(getStoredTeams(release));
    }

    public List<Team> getStoredTeams(Release release) {
        if (release.isArchived()) {
            return release.getTeams();
        } else if (release.getId() != null) {
            String parentId = getParentId(release.getId());
            // Samples and Tutorials still have their own teams and permissions despite the fact
            // that both template and release are in folder
            if (parentId != null && !Ids.isRoot(parentId) && !parentId.equals(TutorialsFolderInitializer.SAMPLES_AND_TUTORIALS_FOLDER_ID())) {
                // folder releases never have their own teams
                return Collections.emptyList();
            }
        }
        return teamRepository.getTeams(securedCis.getSecuredCi(release.getId()));
    }

    public List<Team> getStoredTeams(String containerId) {
        if (isArchivedRelease(containerId)) {
            return archivingService.getRelease(containerId).getTeams();
        } else {
            return teamRepository.getTeams(securedCis.getSecuredCi(containerId));
        }
    }

    private Map<String, Set<String>> getTeamsPermissions(Collection<Team> teams) {
        return teams.stream().collect(Collectors.toMap(
                Team::getTeamName,
                team -> new HashSet<>(team.getPermissions())
        ));
    }

    public boolean isArchivedRelease(String containerId) {
        // releaseRepo.exists check is first here because we have better control over its indexes
        return Ids.isReleaseId(containerId) && !releaseRepository.exists(containerId) && archivingService.exists(containerId);
    }

    public void generateIdIfNecessary(String containerId, Team team) {
        if (isNullId(team.getId())) {
            team.setId(ciIdService.getUniqueId(Type.valueOf(Team.class), containerId));
        }
    }

    public void generateIdIfNecessary(Role role) {
        if (role != null && (isNullId(role.getId()) || "-1".equals(role.getId()))) {
            role.setId(getName(ciIdService.getUniqueId("Role", GLOBAL_ROLES_ROOT)));
        }
    }

    // scala interop
    @Override
    public TeamRepository teamRepository() {
        return this.teamRepository;
    }

    @Override
    public CiIdService ciIdService() {
        return this.ciIdService;
    }

    @Override
    public SecurityRepository securityRepository() {
        return this.securityRepository;
    }

    @Override
    public SecuredCis securedCis() {
        return this.securedCis;
    }

    @Override
    public XLReleaseEventBus eventBus() {
        return this.eventBus;
    }
}
