package com.xebialabs.xlrelease.events;

import java.lang.reflect.InvocationTargetException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.stereotype.Component;

import com.xebialabs.xlrelease.domain.events.XLReleaseEvent;
import com.xebialabs.xlrelease.user.User;
import com.xebialabs.xlrelease.validation.XlrValidationsFailedException;

import net.engio.mbassy.bus.IMessagePublication;
import net.engio.mbassy.bus.MBassador;
import net.engio.mbassy.bus.error.PublicationError;

/**
 * Digital.ai Release specific event bus.
 */
@Component
public class XLReleaseEventBus implements EventBus, DisposableBean {

    private static final Logger logger = LoggerFactory.getLogger(XLReleaseEventBus.class);

    private MBassador bus = new MBassador(this::handleError);
    private volatile boolean enabled = true;

    public void register(Object listener) {
        bus.subscribe(listener);
    }

    public void deregister(Object listener) {
        bus.unsubscribe(listener);
    }

    @SuppressWarnings("unchecked")
    public void publish(XLReleaseEvent event) {
        if (enabled) {
            fillUsername(event);
            logger.debug("Publishing event [{}]", event);
            bus.publish(event);
        } else {
            logger.debug("Not publishing event [{}]", event);
        }
    }

    public void start() {
        enabled = true;
    }

    public void stop() {
        enabled = false;
    }

    @SuppressWarnings("unchecked")
    public void publishAndFailOnError(XLReleaseEvent event) {
        fillUsername(event);

        IMessagePublication pub = bus.publish(event);
        if (pub.hasError()) {
            final Throwable cause = pub.getError().getCause();
            if (cause instanceof InvocationTargetException) {
                Throwable target = ((InvocationTargetException) cause).getTargetException();
                if (target instanceof RuntimeException) {
                    throw (RuntimeException) target;
                }
            }
            if (cause instanceof RuntimeException) {
                throw (RuntimeException) cause;
            }
        }
    }

    @Override
    public boolean hasPendingMessages() {
        return bus.hasPendingMessages();
    }

    private void fillUsername(XLReleaseEvent event) {
        String userName = User.AUTHENTICATED_USER.getName();
        event.username_$eq(userName != null ? userName : "SYSTEM");
    }

    private void handleError(PublicationError error) {
        if (null != error.getCause() && error.getCause().getCause() instanceof XlrValidationsFailedException) {
            logger.error(error.getCause().getCause().getMessage());
        } else if (null != error.getCause()) {
            logger.error(error.getMessage(), error.getCause());
        } else {
            logger.error(error.toString());
        }
    }

    @Override
    public void destroy() {
        try {
            logger.info("Shutting down Release event bus");
            bus.shutdown();
        } catch (Exception e) {
            logger.error("Unable to shutdown event bus", e);
        }
    }
}
