package com.xebialabs.xlrelease.scheduler.workers

import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.domain.CustomScriptTask
import com.xebialabs.xlrelease.scheduler.workers.Worker.{CustomScriptTaskExecutionResult, ExecuteJob, ProcessJobResult}
import com.xebialabs.xlrelease.scheduler.{CustomScriptTaskJob, NextCustomScriptTaskJob}
import com.xebialabs.xlrelease.script._
import com.xebialabs.xlrelease.service.{CommentService, PhaseService}
import com.xebialabs.xlrelease.user.User.{AUTHENTICATED_USER, SYSTEM}
import grizzled.slf4j.Logging
import org.springframework.stereotype.Component

import scala.util.{Failure, Success, Try}


@Component
class CustomScriptTaskWorker(val releaseActorService: ReleaseActorService,
                             phaseService: PhaseService,
                             val scriptServicesList: java.util.List[ScriptService],
                             commentService: CommentService
                            )
  extends Worker with ScriptServiceSupport with TaskWorkerFailureLogic with Logging {

  override def execute: ExecuteJob = {
    case CustomScriptTaskJob(taskRef) => doExecute(taskRef)
    case NextCustomScriptTaskJob(taskRef) => doExecute(taskRef)
  }

  private def doExecute(taskRef: TaskSoftReference[CustomScriptTask]): CustomScriptTaskExecutionResult = {
    val task = taskRef.get()
    val result = Try {
      scriptServiceFor(task).executeCustomScriptTask(task)
    }
    CustomScriptTaskExecutionResult(task.getId, task.getExecutionId, result)
  }

  override def processResult: ProcessJobResult = {
    case CustomScriptTaskExecutionResult(taskId, executionId, executionResult) =>
      executionResult match {
        case Failure(error) => handleFailure(taskId, executionId, error)
        case Success(result) => handleCustomScriptTaskResult(result)
      }
  }

  def handleCustomScriptTaskResult(result: CustomScriptTaskResult): Unit = {
    result match {
      case SuccessCustomScriptTaskResult(taskId, executionId, executionLog, attachmentId, results, authentication) =>
        withAuthentication(authentication) {
          val nextScriptPath = results.getNextScriptPath
          if (nextScriptPath != null) {
            releaseActorService.saveCustomScriptResults(taskId, results, executionId)
          } else {
            releaseActorService.finishCustomScriptTask(taskId, executionId, executionLog, attachmentId, Some(results))
          }
        }
      case FailureCustomScriptTaskResult(taskId, executionId, executionLog, attachmentId, authentication, results) =>
        withAuthentication(authentication) {
          releaseActorService.failScriptTask(taskId, executionLog, executionId, attachmentId, results)
        }
      case RestartCustomScriptTaskResult(taskId, executionId, _, _, exception, authentication, results) =>
        withAuthentication(authentication) {
          commentService.create(taskId, exception.getMessage, SYSTEM, false)
          releaseActorService.failScriptTask(taskId, null, executionId, None, results)
          try releaseActorService.restartPhase(
            exception.getReleaseId, exception.getPhaseId, exception.getTaskId, exception.getPhaseVersion, exception.isResumeRelease
          )
          catch {
            case e: Exception =>
              commentService.create(taskId, e.getMessage, AUTHENTICATED_USER, true);
          }
        }
    }
  }
}

