package com.xebialabs.xlrelease.security

import com.xebialabs.deployit.checks.Checks
import com.xebialabs.deployit.security.{Permissions, Role, RoleService}
import com.xebialabs.xlrelease.domain.Team
import com.xebialabs.xlrelease.repository.Ids.{isFolderId, isReleaseId}
import com.xebialabs.xlrelease.security.PermissionChecker.{folderAdminCannotAssignPermissionException, folderAdminCannotDeleteTeamException, folderAdminCannotEditTeamException, userCannotManageTeamException}
import com.xebialabs.xlrelease.security.XLReleasePermissions._
import com.xebialabs.xlrelease.security.{PermissionChecker => XlrPermissionChecker}
import com.xebialabs.xlrelease.service._
import com.xebialabs.xlrelease.user.User

import java.util
import java.util.stream.Collectors.toList
import java.util.{List => JList}
import scala.jdk.CollectionConverters._

trait PermissionCheckerSharedLogic {
  self: XlrPermissionChecker =>

  def teamService: TeamService

  def roleService: RoleService

  def checkEditTeamsAgainstExisting(teamContainerId: String, teams: util.Collection[Team]): Unit = {
    if (isReleaseId(teamContainerId)) {
      checkEdit(teamContainerId)
    } else if (isFolderId(teamContainerId)) {
      checkEditFolderTeamsAgainstExisting(teamContainerId, teams.asScala.toSeq)
    } else {
      throw new Checks.IncorrectArgumentException("[%s] is not a release or a folder", teamContainerId)
    }
  }

  // scalastyle:off cyclomatic.complexity
  private def checkEditFolderTeamsAgainstExisting(teamContainerId: String, teams: Seq[Team]): Unit = {
    val superUser: Boolean = isCurrentUserAdmin
    val folderSecurityAdministrator: Boolean = hasPermission(EDIT_FOLDER_SECURITY, teamContainerId)
    val folderTeamAdmin: Boolean = hasPermission(EDIT_FOLDER_TEAMS, teamContainerId)
    if (folderTeamAdmin && !folderSecurityAdministrator && !superUser) {
      val folderTeams = teamService.getEffectiveTeams(teamContainerId).asScala
      // Team implements deep equals and filter covers all teams that are not same as existing folder teams
      val newAndUpdatedTeams: Seq[Team] = teams.filter(!folderTeams.contains(_))
      newAndUpdatedTeams.foreach { team =>
        if (team.isFolderOwnerTeam || team.isFolderAdminTeam) {
          throw XlrPermissionChecker.folderAdminCannotAssignPermissionException
        }
        if (team.hasPermission(EDIT_FOLDER.getPermissionName)) {
          throw XlrPermissionChecker.folderAdminCannotAssignPermissionException
        }
        val oldTeam: Option[Team] = folderTeams.find(_.getTeamName == team.getTeamName)
        oldTeam.foreach((old: Team) => {
          if (old.isFolderOwnerTeam || old.isFolderAdminTeam) {
            throw XlrPermissionChecker.folderAdminCannotEditTeamException
          }
        })
      }
    } else if (!superUser && !folderSecurityAdministrator) {
      throw XlrPermissionChecker.userCannotManageTeamException
    }
  }
  // scalastyle:on cyclomatic.complexity

  def checkDeleteOwnTeams(teamContainerId: String): Unit = {
    if (isReleaseId(teamContainerId)) {
      checkEdit(teamContainerId)
    } else if (isFolderId(teamContainerId)) {
      val superUser: Boolean = isCurrentUserAdmin
      val owner: Boolean = hasPermission(EDIT_FOLDER_SECURITY, teamContainerId)
      val admin: Boolean = hasPermission(EDIT_FOLDER_TEAMS, teamContainerId)
      if (admin && !owner && !superUser) {
        val userFolderTeams = getUserFolderTeams(teamContainerId).asScala
        userFolderTeams.foreach { team =>
          if (team.isFolderOwnerTeam || team.isFolderAdminTeam) {
            throw XlrPermissionChecker.folderAdminCannotDeleteTeamException
          }
        }
      } else if (!superUser && !owner) {
        throw XlrPermissionChecker.userCannotManageTeamException
      }
    } else {
      throw new Checks.IncorrectArgumentException("[%s] is not a release or a folder", teamContainerId)
    }
  }

  def getUserFolderTeams(folderId: String): JList[Team] = {
    val username: String = User.AUTHENTICATED_USER.getName
    val folderTeams: JList[Team] = teamService.getEffectiveTeams(folderId)
    val userRoles: JList[Role] = roleService.getRolesFor(Permissions.getAuthentication)
    folderTeams.stream.filter((team: Team) => team.hasMember(username) || team.hasAnyRole(userRoles)).collect(toList())
  }

  def checkEditTeams(teamContainerId: String, operations: Seq[_ <: TeamUpdateOperation]): Unit = {
    if (isReleaseId(teamContainerId)) {
      checkEdit(teamContainerId)
    } else if (isFolderId(teamContainerId)) {
      checkEditFolderTeams(teamContainerId, operations)
    } else {
      throw new Checks.IncorrectArgumentException("[%s] is not a release or a folder", teamContainerId)
    }
  }

  def checkEditTeamMembers(teamContainerId: String, team: Team): Unit = {
    val superUser: Boolean = isCurrentUserAdmin
    val folderOwner: Boolean = hasPermission(EDIT_FOLDER_SECURITY, teamContainerId)
    val folderAdmin: Boolean = hasPermission(EDIT_FOLDER_TEAMS, teamContainerId)

    if (folderAdmin && !folderOwner && !superUser) {
      if (team.isFolderOwnerTeam || team.isFolderAdminTeam || team.hasPermission(EDIT_FOLDER.getPermissionName)) {
        throw folderAdminCannotDeleteTeamException
      }
    } else if (!superUser && !folderOwner) {
      throw userCannotManageTeamException
    }
  }

  def checkDeleteTeam(teamContainerId: String, team: Team): Unit = {
    val superUser: Boolean = isCurrentUserAdmin
    val folderOwner: Boolean = hasPermission(EDIT_FOLDER_SECURITY, teamContainerId)
    val folderAdmin: Boolean = hasPermission(EDIT_FOLDER_TEAMS, teamContainerId)

    if (folderAdmin && !folderOwner && !superUser) {
      if (team.isFolderOwnerTeam || team.isFolderAdminTeam) {
        throw folderAdminCannotDeleteTeamException
      }
    } else if (!superUser && !folderOwner) {
      throw userCannotManageTeamException
    }
  }

  def checkEditTeamPermissions(teamContainerId: String, addedPermissions: Seq[String] = Seq.empty, removedPermissions: Seq[String] = Seq.empty): Unit = {
    val superUser: Boolean = isCurrentUserAdmin
    val folderOwner: Boolean = hasPermission(EDIT_FOLDER_SECURITY, teamContainerId)
    val folderAdmin: Boolean = hasPermission(EDIT_FOLDER_TEAMS, teamContainerId)

    if (folderAdmin && !folderOwner && !superUser) {
      val folderOwnerAdded = addedPermissions.contains(EDIT_FOLDER_SECURITY.getPermissionName)
      val folderAdminAdded = addedPermissions.contains(EDIT_FOLDER_TEAMS.getPermissionName)
      if (folderOwnerAdded || folderAdminAdded) {
        // Modifed team state contains permissions EDIT_FOLDER_SECURITY or EDIT_FOLDER_TEAMS
        throw folderAdminCannotAssignPermissionException
      }
      if (addedPermissions.contains(EDIT_FOLDER.getPermissionName)) {
        throw folderAdminCannotAssignPermissionException
      }
      val permissionsToCheck = Set(EDIT_FOLDER_SECURITY.getPermissionName, EDIT_FOLDER_TEAMS.getPermissionName)
      if (removedPermissions.toSet.intersect(permissionsToCheck).nonEmpty) {
        throw folderAdminCannotEditTeamException
      }
    } else if (!superUser && !folderOwner) {
      throw userCannotManageTeamException
    }
  }

  private def checkEditFolderTeams(teamContainerId: String, operations: Seq[TeamUpdateOperation]): Unit = {
    def checkEditOperation(teamOperation: TeamUpdateOperation): Unit = {
      if (teamOperation.isFolderOwnerTeam || teamOperation.isFolderAdminTeam) {
        throw folderAdminCannotAssignPermissionException
      }
      if (teamOperation.hasPermission(EDIT_FOLDER.getPermissionName)) {
        throw folderAdminCannotAssignPermissionException
      }
    }

    def checkCreate(teamOperation: CreateTeamOperation): Unit = {
      checkEditOperation(teamOperation)
    }

    def checkUpdate(teamOperation: UpdateTeamOperation): Unit = {
      checkEditOperation(teamOperation)
      val removedMembership = teamOperation.removedMembership
      // should not remove folderOwner and folderAdmin permissions
      val permissionsToCheck = Set(EDIT_FOLDER_SECURITY.getPermissionName, EDIT_FOLDER_TEAMS.getPermissionName)
      if (removedMembership.permissions.intersect(permissionsToCheck).nonEmpty) {
        throw folderAdminCannotEditTeamException
      }
    }

    def checkDelete(teamOperation: DeleteTeamOperation): Unit = {
      val maybeTeam = teamService.securityRepository().findTeamByName(teamOperation.containerId, teamOperation.team)

      maybeTeam.foreach(team => {
        if (team.isFolderOwnerTeam || team.isFolderAdminTeam) {
          throw XlrPermissionChecker.folderAdminCannotDeleteTeamException
        }
      })
    }

    val superUser: Boolean = isCurrentUserAdmin
    val folderOwner: Boolean = hasPermission(EDIT_FOLDER_SECURITY, teamContainerId)
    val folderAdmin: Boolean = hasPermission(EDIT_FOLDER_TEAMS, teamContainerId)
    if (folderAdmin && !folderOwner && !superUser) {
      operations.foreach {
        case _: BreakInheritanceOperation =>
        case create: CreateTeamOperation => checkCreate(create)
        case update: UpdateTeamOperation => checkUpdate(update)
        case delete: DeleteTeamOperation => checkDelete(delete)
      }
    } else if (!superUser && !folderOwner) {
      throw userCannotManageTeamException
    }
  }


}
