package com.xebialabs.xlrelease.db

import com.xebialabs.xlrelease.XLReleaseServerLaunchOptions
import com.xebialabs.xlrelease.config.XlrConfig
import grizzled.slf4j.Logging
import liquibase.exception.LiquibaseException
import liquibase.snapshot.SnapshotGeneratorFactory
import liquibase.{Contexts, LabelExpression}
import org.springframework.beans.factory.annotation.Autowired

import javax.sql.DataSource

class XLReleaseDbInitializer(val dbConfig: String, val dataSource: DataSource, xlrConfig: XlrConfig) extends LiquibaseSupport with Logging {

  @Autowired
  var launchOptions: XLReleaseServerLaunchOptions = _

  def run(): Unit = {
    xlrConfig.upgraderMode match {
      case xlrConfig.upgrader.VERIFY_MODE =>
        try {
          verify()
        } catch {
          case ex: LiquibaseException =>
            logger.error("Liquibase verification failed: ", ex)
            throw new RuntimeException("Upgrade verification failed, run upgrader in apply mode.");
        }

      case xlrConfig.upgrader.APPLY_MODE =>
        update()
    }
  }

  def update(): Unit = {
    doWithLiquibase(dbConfig) { liquibase =>
      if (launchOptions != null && launchOptions.isReleaseDbLocks) {
        logger.info("Releasing liquibase  database locks")
        liquibase.forceReleaseLocks()
      }
      // TODO replace the deprecated, Liquibase executes change sets based on Context cache hence the change
      liquibase.update(new Contexts(dbConfig))
    }
  }

  def verify(): Unit = {
    doWithLiquibase(dbConfig) { liquibase =>
      liquibase.validate()
      val unrunChangesets = liquibase.listUnrunChangeSets(new Contexts(dbConfig), new LabelExpression)
      if (!unrunChangesets.isEmpty) {
        throw new LiquibaseException("Missing liquibase changesets found.")
      }
    }
  }

  def isInitialized: Boolean = {
    doWithLiquibase(dbConfig) { liquibase =>
      SnapshotGeneratorFactory.getInstance().hasDatabaseChangeLogTable(liquibase.getDatabase)
    }
  }

}
