package com.xebialabs.xlrelease.actors

import com.xebialabs.xlrelease.actors.BroadcastEventActor.{PublishEvent, PublishEventRequested}
import com.xebialabs.xlrelease.domain.events.XLReleaseEvent
import com.xebialabs.xlrelease.events.EventBus
import com.xebialabs.xlrelease.support.pekko.spring.SpringActor
import org.apache.pekko.actor.{Actor, ActorLogging, ActorRef}
import org.apache.pekko.cluster.pubsub.DistributedPubSub
import org.apache.pekko.cluster.pubsub.DistributedPubSubMediator.{Publish, Subscribe, SubscribeAck}

import scala.util.Try

object BroadcastEventActor {

  val name = "broadcastEventActor"

  sealed trait BroadcastEventCommand

  case class PublishEvent(event: XLReleaseEvent, publishEventOnSelf: Boolean) extends BroadcastEventCommand

  case class PublishEventRequested(event: XLReleaseEvent, publishEventOnSelf: Boolean) extends BroadcastEventCommand

}

@SpringActor
class BroadcastEventActor(eventBus: EventBus) extends Actor with ActorLogging {

  val topicName = "XlrEventBroadcastActor"
  val mediator: ActorRef = DistributedPubSub(context.system).mediator
  mediator ! Subscribe(topicName, self)

  override def receive: Receive = {
    case SubscribeAck(Subscribe(topicName, _, _)) =>
      log.info(s"Subscribed to topic '$topicName'")
    case PublishEvent(event, publishEventOnSelf) =>
        mediator ! Publish(topicName, PublishEventRequested(event, publishEventOnSelf))
    case PublishEventRequested(event, publishEventOnSelf) =>
      if (publishEventOnSelf || sender() != self) {
        log.debug(s"Publishing event '$event' to local event bus")
        Try(eventBus.publish(event)).recover {
          case e: Throwable => log.error(e, s"Unable to publish event $event to local event bus")
        }
      } else {
        log.debug(s"Ignoring event '$event' and not publishing to local event bus")
      }
  }
}

