package com.xebialabs.xlrelease.actors

import com.xebialabs.xlrelease.actors.NonShardedReleasesActor.bufferedMessages
import com.xebialabs.xlrelease.actors.ReleaseSupervisorActor.{GracefulPoisonPill, NonShardedPassivate}
import com.xebialabs.xlrelease.actors.sharding.ReleaseShardingMessages.ReleaseAction
import com.xebialabs.xlrelease.config.XlrConfig
import org.apache.pekko.actor.{Actor, ActorLogging, ActorRef, Props}
import org.apache.pekko.pattern.{GracefulStopSupport, pipe}

import scala.collection.mutable
import scala.collection.mutable.ListBuffer
import scala.concurrent.ExecutionContext

object NonShardedReleasesActor {
  val name = "releases"

  def props(releaseSupervisorProps: Props, xlrConfig: XlrConfig) = Props(new NonShardedReleasesActor(releaseSupervisorProps, xlrConfig))

  private val bufferedMessages: mutable.Map[String, ListBuffer[(ActorRef, AnyRef)]] = mutable.Map()
}

case class GracefulTerminationCompletedOrTimedOut(terminated: Boolean, actorRef: ActorRef, releaseId: String)

class NonShardedReleasesActor(releaseSupervisorProps: Props, xlrConfig: XlrConfig) extends Actor with ActorLogging with GracefulStopSupport{


  override def preStart(): Unit = {
    super.preStart()
    bufferedMessages.clear()
  }

  override def receive: Receive = {
    case releaseAction@ReleaseAction(releaseId, msg) if bufferedMessages.contains(releaseId2ActorName(releaseId)) =>
      log.debug("Buffering message for {}, {}", releaseId, msg)
      bufferedMessages(releaseId2ActorName(releaseId)).addOne((sender(), releaseAction))
    case ReleaseAction(releaseId, msg) =>
      log.debug("forwarding message to {}, {}", releaseId, msg)
      createOrFind(releaseId, msg) forward msg
    case NonShardedPassivate(releaseId) =>
      implicit val ec: ExecutionContext = context.dispatcher
      if (!bufferedMessages.contains(releaseId2ActorName(releaseId))) {
        log.debug("got NonShardedPassivate from {}, creating bufferedMessages entry", releaseId)
        bufferedMessages.put(releaseId2ActorName(releaseId), ListBuffer())
      } else {
        log.debug("got NonShardedPassivate from {}, bufferedMessages entry already exist", releaseId)
      }
      val childActor = sender()
      gracefulStop(childActor, xlrConfig.timeouts.releaseActionResponse, GracefulPoisonPill).recover(e => {
        log.error(e, "Unable to stop ReleaseActor for {} within timeout", releaseId)
        false
      }).map(terminated => GracefulTerminationCompletedOrTimedOut(terminated, childActor, releaseId)).pipeTo(self)
    case msg@GracefulTerminationCompletedOrTimedOut(terminated, actorRef, releaseId) =>
      if (!terminated) {
        log.error("Terminating child actor {} which was unable to gracefully terminate", releaseId)
        context.stop(actorRef)
      }
      if (bufferedMessages.contains(actorRef.path.name)) {
        if (bufferedMessages(actorRef.path.name).nonEmpty) {
          log.debug("got GracefulTerminationCompletedOrTimedOut, redelivering {} messages to {} after passivation", bufferedMessages(actorRef.path.name).size, releaseId)
          // fine, we have some buffered messages, we need to resend them.
          bufferedMessages(actorRef.path.name).foreach {
            case (origin: ActorRef, msg: AnyRef) => self.tell(msg, origin)
          }
        } else {
          log.debug("got GracefulTerminationCompletedOrTimedOut, no buffered messages to redeliver to {} after passivation", releaseId)
        }
        bufferedMessages.remove(actorRef.path.name)
      } else {
        log.error("got GracefulTerminationCompletedOrTimedOut {} from {}, but bufferedMessages entry does not exist", msg, releaseId)
      }
  }

  private def createOrFind(releaseId: String, msg: AnyRef): ActorRef = {
    context.child(releaseId2ActorName(releaseId)).getOrElse({
      log.debug(s"NonShardedReleasesActor - Creating an actor tree for $releaseId (triggered by $msg)")
      context.actorOf(releaseSupervisorProps, releaseId2ActorName(releaseId))
    })
  }

}
