package com.xebialabs.xlrelease.db.sql

import com.xebialabs.xlplatform.utils.ResourceManagement._

import java.sql.DatabaseMetaData
import javax.sql.DataSource


sealed trait DatabaseInfo {
  def metadata: DatabaseMetaData

  val dbName: String = if (null == metadata) "unknown" else metadata.getDatabaseProductName.toLowerCase

  val dbVersion: String = if (null == metadata) "unknown" else metadata.getDatabaseProductVersion

  override def toString: String = s"$dbName:$dbVersion"
}

object DatabaseInfo {

  def apply(ds: DataSource): DatabaseInfo = {
    map(ds.getConnection) { connection =>
      val metadata = connection.getMetaData
      DatabaseInfo(metadata)
    }
  }

  def apply(metadata: DatabaseMetaData): DatabaseInfo = {
    val dbName = metadata.getDatabaseProductName
    val dbNameLower: String = dbName.toLowerCase
    if (dbNameLower.contains("db2")) {
      Db2(metadata)
    } else if (dbNameLower.contains("derby")) {
      Derby(metadata)
    } else if (dbNameLower.contains("h2")) {
      H2(metadata)
    } else if (dbNameLower.contains("sql server")) {
      MsSqlServer(metadata)
    } else if (dbNameLower.contains("mysql")) {
      MySql(metadata)
    } else if (dbNameLower.contains("oracle")) {
      Oracle(metadata)
    } else if (dbNameLower.contains("postgresql")) {
      PostgreSql(metadata)
    } else {
      Unknown(metadata)
    }
  }

  case class TestDatabaseInfo(metadata: DatabaseMetaData) extends DatabaseInfo // only for unit tests

  case class Unknown(metadata: DatabaseMetaData) extends DatabaseInfo

  case class Db2(metadata: DatabaseMetaData) extends DatabaseInfo

  case class Derby(metadata: DatabaseMetaData) extends DatabaseInfo

  case class H2(metadata: DatabaseMetaData) extends DatabaseInfo

  case class MsSqlServer(metadata: DatabaseMetaData) extends DatabaseInfo

  case class MySql(metadata: DatabaseMetaData) extends DatabaseInfo

  case class Oracle(metadata: DatabaseMetaData) extends DatabaseInfo

  case class PostgreSql(metadata: DatabaseMetaData) extends DatabaseInfo

}