package com.xebialabs.xlrelease.domain.tasks.createrelease;

import java.util.List;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xlrelease.domain.BaseConfiguration;
import com.xebialabs.xlrelease.domain.CreateReleaseTask;
import com.xebialabs.xlrelease.repository.ConfigurationRepository;
import com.xebialabs.xlrelease.repository.Ids;
import com.xebialabs.xlrelease.service.ReleaseSearchService;
import com.xebialabs.xlrelease.variable.VariableHelper;
import com.xebialabs.xlrelease.views.TaskFullView;
import com.xebialabs.xlrelease.views.converters.BaseTaskViewConverter;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.xlrelease.builder.TaskBuilder.newCreateReleaseTask;
import static com.xebialabs.xlrelease.repository.IdType.DOMAIN;
import static com.xebialabs.xlrelease.risk.domain.RiskProfile.RISK_PROFILE;
import static org.springframework.util.StringUtils.hasText;

@Component
public class CreateReleaseTaskViewConverter extends BaseTaskViewConverter<CreateReleaseTask> {

    private final ConfigurationRepository configurationRepository;

    private final ReleaseSearchService releaseSearchService;

    @Autowired
    public CreateReleaseTaskViewConverter(final ConfigurationRepository configurationRepository, ReleaseSearchService releaseSearchService) {
        this.configurationRepository = configurationRepository;
        this.releaseSearchService = releaseSearchService;
    }

    @Override
    public TaskFullView toFullView(final CreateReleaseTask task, final List<Type> allowedTaskTypesForAuthenticatedUser) {
        TaskFullView v = super.toFullView(task, allowedTaskTypesForAuthenticatedUser);
        v.setNewReleaseTitle(task.getNewReleaseTitle());
        v.setTemplateId(task.getTemplateId());
        v.setFolderId(task.getFolderId());
        v.setVariables(task.getTemplateVariables());
        final String createdReleaseId = task.getCreatedReleaseId();
        v.setCreatedReleaseId(createdReleaseId);
        if (StringUtils.hasText(createdReleaseId) && Ids.isReleaseId(createdReleaseId)) {
            releaseSearchService.getTitle(createdReleaseId).foreach(t -> {
                v.setCreatedReleaseTitle(t);
                return null;
            });
        }
        v.setStartRelease(task.getStartRelease());
        v.setWaitForRelease(task.getWaitForRelease());
        v.setReleaseTags(task.getReleaseTags());
        if (task.hasProperty(RISK_PROFILE)) {
            setRiskProfileFromTask(task, v);
        }
        return v;
    }

    private void setRiskProfileFromTask(final CreateReleaseTask task, final TaskFullView view) {
        BaseConfiguration configuration = task.getProperty(RISK_PROFILE);
        if (null != configuration && hasText(configuration.getId())) {
            view.setRiskProfileId(configuration.getId());
        }
    }

    @Override
    protected CreateReleaseTask fromView(final TaskFullView view) {
        VariableHelper.collectVariables(view.getTemplateId()).forEach(view::setTemplateId);
        VariableHelper.collectVariables(view.getFolderId()).forEach(view::setFolderId);
        CreateReleaseTask task = newCreateReleaseTask()
                .withNewReleaseTitle(view.getNewReleaseTitle())
                .withTemplateId(view.getTemplateId())
                .withFolderId(view.getFolderId())
                .withVariables(view.getVariables())
                .withCreatedReleaseId(view.getCreatedReleaseId())
                .withStartRelease(view.getStartRelease())
                .withWaitForRelease(view.getWaitForRelease())
                .withReleaseTags(view.getReleaseTags())
                .build();
        if (task.hasProperty(RISK_PROFILE)) {
            setRiskProfileFromView(view, task);
        }
        return task;
    }

    private void setRiskProfileFromView(final TaskFullView view, final CreateReleaseTask task) {
        String configurationId = view.getRiskProfileId();
        if (hasText(configurationId)) {
            BaseConfiguration configuration = configurationRepository.read(configurationId);
            PropertyDescriptor riskProfilePD = task.getType().getDescriptor().getPropertyDescriptor(RISK_PROFILE);
            checkArgument(configuration.getType().getDescriptor().isAssignableTo(riskProfilePD.getReferencedType()),
                    "Can't assign property of type " + configuration.getType().getName() + " to property " + riskProfilePD.getName());
            task.setProperty(RISK_PROFILE, configuration);
        }
    }

    @Override
    protected Class<CreateReleaseTask> getTaskClass() {
        return CreateReleaseTask.class;
    }
}
