package com.xebialabs.xlrelease.repository

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.api.v1.forms.FacetFilters
import com.xebialabs.xlrelease.domain.facet.{Facet, TaskReportingRecord}
import com.xebialabs.xlrelease.domain.{Release, Task}
import com.xebialabs.xlrelease.repository.Ids.releaseIdFrom
import com.xebialabs.xlrelease.service.CiIdService

import java.util.Date
import scala.collection.mutable
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Try}

trait FacetRepository {

  def ciIdService: CiIdService


  def get(facetId: String, facetType: Option[Type] = None): Facet

  protected def doCreate(facet: Facet): Facet

  protected def setCreationDate(facet: Facet): Unit = ()

  protected def generateId(facet: Facet): String = {
    Ids.getName(ciIdService.getUniqueId(Type.valueOf(classOf[Facet]), ""))
  }

  def create(facet: Facet): Facet = {
    if (facet.getId == null) {
      facet.setId(generateId(facet))
    }
    setCreationDate(facet)
    doCreate(facet)
  }

  def createFromTasks(tasks: Seq[Task], filter: Facet => Boolean = _ => true): Unit = {
    for {
      task <- tasks
      facet <- task.getFacets.asScala
      if filter(facet)
    } {
      facet.setId(generateId(facet))
      facet.setTargetId(task.getId)
      create(facet)
    }
  }

  def delete(facetId: String, facetType: Option[Type] = None): Unit

  def update(facet: Facet): Facet

  def search(facetsFilters: FacetFilters): Seq[Facet]

  def exists(facetId: String, facetType: Option[Type] = None): Boolean

  def findAllFacetsByRelease(release: Release): Seq[Facet]

  def findAllFacetsByTask(task: Task): Seq[Facet]

}

object FacetRepository {

  protected def taskReportingRecordType: Type = Type.valueOf(classOf[TaskReportingRecord])

  trait GenericFacetRepository extends FacetRepository

  abstract class SpecializedFacetRepository(val supportedType: Type) extends FacetRepository

  trait ForLive { self: FacetRepository =>
    override def setCreationDate(facet: Facet): Unit = {
      if (facet.getType.instanceOf(Type.valueOf(classOf[TaskReportingRecord]))) {
        facet.asInstanceOf[TaskReportingRecord].setCreationDate(new Date)
      }
    }
  }
  trait ForArchive { self: FacetRepository => }

  trait FacetTypeDispatcher {
    self: FacetRepository =>

    val genericRepository: GenericFacetRepository
    val specializedRepositories: mutable.Set[SpecializedFacetRepository] = mutable.Set.empty

    def register(repository: SpecializedFacetRepository): Unit = {
      require(repository.supportedType.instanceOf(taskReportingRecordType))
      specializedRepositories += repository
    }

    def get(facetId: String, facetType: Option[Type] = None): Facet =
      repositoryByType(facetType).get(facetId, facetType)

    override protected def doCreate(facet: Facet): Facet =
      repositoryByType(facet.getType).doCreate(facet)

    def delete(facetId: String, facetType: Option[Type] = None): Unit =
      repositoryByType(facetType).delete(facetId, facetType)

    def update(facet: Facet): Facet =
      repositoryByType(facet.getType).update(facet)

    def search(facetsFilters: FacetFilters): Seq[Facet] = {
      val repositoriesForThisSearch =
        if (facetsFilters.getTypes == null || facetsFilters.getTypes.isEmpty) {
          repositories
        } else {
          facetsFilters.getTypes.asScala
            .map(facetType => repositoryByType(facetType))
            .distinct
        }
      repositoriesForThisSearch.flatMap(_.search(facetsFilters)).toSeq
    }

    def exists(facetId: String, facetType: Option[Type] = None): Boolean =
      repositoryByType(facetType).exists(facetId, facetType)

    def findAllFacetsByRelease(release: Release): Seq[Facet] =
      repositories.flatMap(_.findAllFacetsByRelease(release))

    def findAllFacetsByTask(task: Task): Seq[Facet] =
      repositories.flatMap(_.findAllFacetsByTask(task))

    def repositoryByType(facetType: Type): FacetRepository =
      specializedRepositories.find(specialized => facetType.instanceOf(specialized.supportedType)) match {
        case None => genericRepository
        case Some(specialized) => specialized
      }

    protected def repositoryByType(facetType: Option[Type]): FacetRepository =
      facetType.map(repositoryByType).getOrElse(genericRepository)

    protected def repositories: Seq[FacetRepository] = genericRepository +: specializedRepositories.toSeq

    protected def dispatch[A](facetType: Type)(f: FacetRepository => A): A =
      f(repositoryByType(facetType))
  }

  trait DatabaseDispatcher[R <: FacetRepository] {
    self: R =>

    def liveRepository: R with ForLive
    def archiveRepository: R with ForArchive

    def isArchived(releaseId: String): Boolean

    def get(facetId: String, facetType: Option[Type] = None): Facet =
      tryLiveThenArchive(_.get(facetId, facetType))

    override protected def doCreate(facet: Facet): Facet =
      dispatch(releaseIdFrom(facet.getTargetId))(_.create(facet))

    def delete(facetId: String, facetType: Option[Type] = None): Unit = {
      liveRepository.delete(facetId, facetType)
    }

    def update(facet: Facet): Facet =
      dispatch(releaseIdFrom(facet.getTargetId))(_.update(facet))

    def search(facetsFilters: FacetFilters): Seq[Facet] =
      liveRepository.search(facetsFilters) ++ archiveRepository.search(facetsFilters)

    def exists(facetId: String, facetType: Option[Type] = None): Boolean =
      liveRepository.exists(facetId, facetType) || archiveRepository.exists(facetId, facetType)

    // TODO: determine which database, then issue single query
    def findAllFacetsByRelease(release: Release): Seq[Facet] = {
      liveRepository.findAllFacetsByRelease(release) ++ archiveRepository.findAllFacetsByRelease(release)
    }

    // TODO: determine which database, then issue single query
    def findAllFacetsByTask(task: Task): Seq[Facet] =
      liveRepository.findAllFacetsByTask(task) ++ archiveRepository.findAllFacetsByTask(task)

    private def dispatch[A](releaseId: String)(f: FacetRepository => A): A =
      if (isArchived(releaseId)) {
        f(archiveRepository)
      } else {
        f(liveRepository)
      }

    private def tryLiveThenArchive[A](f: FacetRepository => A): A =
      Try(f(liveRepository)).recoverWith {
        case orig: NotFoundException =>
          Try(f(archiveRepository)).recoverWith {
            case nfe: NotFoundException => Failure(nfe)
            case _ => Failure(orig)
          }
      }.get

  }
}
