package com.xebialabs.xlrelease.scheduler.strategies

import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.scheduler.strategies.JobSchedulerStrategy.ScheduleResult
import com.xebialabs.xlrelease.scheduler.{Job, TaskJob}
import com.xebialabs.xlrelease.support.pekko.spring.ScalaSpringAwareBean
import grizzled.slf4j.Logging
import org.apache.pekko.util.Timeout

import scala.concurrent.{Await, TimeoutException}
import scala.util.{Failure, Success, Try}

object NonBlockingBackpressuredJobSchedulerStrategy extends JobSchedulerStrategy[NonBlockingBackpressuredJobSchedulerStrategySettings]
  with ScalaSpringAwareBean with Logging {

  lazy val releaseActorService: ReleaseActorService = springBean[ReleaseActorService]
  lazy val config = springBean[XlrConfig]

  override def schedule(configuration: NonBlockingBackpressuredJobSchedulerStrategySettings)(job: Job): ScheduleResult = {
    job match {
      case job: TaskJob[_] =>
        handleJob(configuration, job)
      case j => Right(j)
    }
  }

  private def handleJob(configuration: NonBlockingBackpressuredJobSchedulerStrategySettings, job: TaskJob[_]) = {
    val jobShouldBeDelayed = shouldBeDelayed(configuration, job.taskId, System.currentTimeMillis())
    if (jobShouldBeDelayed) {
      val delay = configuration.delayDuration
      job.delay(delay)
      Left(job)
    } else {
      Right(job)
    }
  }

  private[strategies] def shouldBeDelayed(configuration: NonBlockingBackpressuredJobSchedulerStrategySettings,
                                          taskId: String,
                                          start: Long): Boolean = {
    val attemptStart = System.currentTimeMillis()
    implicit val askTimeout: Timeout = config.timeouts.releaseActionResponse
    Try {
      Await.result(releaseActorService.sendBackpressure(taskId), askTimeout.duration)
    } match {
      case Failure(e: TimeoutException) => ()
      case Failure(e: InterruptedException) =>
        logger.error(s"Got RuntimeException in shouldBeDelayed ${taskId}", e)
        throw new RuntimeException(e)
      case Failure(e: Throwable) =>
        logger.error(s"Got exception in shouldBeDelayed ${taskId}", e)
        throw e
      case Success(response) => ()
    }
    val responseTime = System.currentTimeMillis()
    val taskBackpressureResponseThresholdRatio = configuration.responseThresholdRatio
    val threshold = computeBackpressureThreshold(configuration, start, responseTime, taskBackpressureResponseThresholdRatio)
    if ((responseTime - attemptStart) < threshold) {
      logger.debug(s"Backpressure took ${responseTime - start} ms, ping ${responseTime - attemptStart} ms, threshold ${threshold} ms for ${taskId}")
      false
    } else {
      logger.debug(s"Backpressure waiting ${responseTime - start} ms, ping ${responseTime - attemptStart} ms, threshold ${threshold} ms for ${taskId}")
      true
    }
  }

  private def computeBackpressureThreshold(configuration: NonBlockingBackpressuredJobSchedulerStrategySettings,
                                           start: Long,
                                           responseTime: Long,
                                           ratio: Long) = {
    val taskBackpressureResponseThreshold = configuration.responseThreshold
    taskBackpressureResponseThreshold.toMillis + (responseTime - start) / ratio
  }

}
