package com.xebialabs.xlrelease.scheduler.workers

import com.google.common.base.Preconditions.checkArgument
import com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.builder.ReleaseBuilder.newRelease
import com.xebialabs.xlrelease.domain.events.{CreatedFromCreateReleaseTask, ReleaseCreatedEvent, ReleaseCreatedFromCreateReleaseTaskEvent, ReleaseStartedFromCreateReleaseTaskEvent}
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.domain.{CreateReleaseTask, Release}
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.repository.IdType.DOMAIN
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.risk.domain.RiskProfile.RISK_PROFILE
import com.xebialabs.xlrelease.scheduler.CreateReleaseTaskJob
import com.xebialabs.xlrelease.scheduler.workers.Worker._
import com.xebialabs.xlrelease.script.{EncryptionHelper, TaskSoftReference}
import com.xebialabs.xlrelease.security.XLReleasePermissions.START_RELEASE
import com.xebialabs.xlrelease.security.authentication.AuthenticationService
import com.xebialabs.xlrelease.security.{PermissionChecker, UsernamePassword}
import com.xebialabs.xlrelease.service.{CiIdService, CommentService, ReleaseService}
import com.xebialabs.xlrelease.user.User.{AUTHENTICATED_USER, SYSTEM}
import com.xebialabs.xlrelease.variable.VariableHelper
import com.xebialabs.xlrelease.variable.VariableHelper.fillVariableValues
import com.xebialabs.xlrelease.variable.VariablePersistenceHelper.fixUpVariableIds
import grizzled.slf4j.Logging
import org.springframework.stereotype.Component
import org.springframework.util.StringUtils.hasText

import java.lang.String.format
import java.util.{Optional, ArrayList => JArrayList, List => JList, Map => JMap}
import scala.jdk.OptionConverters._

@Component
class CreateReleaseTaskWorker(val eventBus: XLReleaseEventBus,
                              val releaseActorService: ReleaseActorService,
                              val releaseService: ReleaseService,
                              val ciIdService: CiIdService,
                              val authenticationService: AuthenticationService,
                              val permissionChecker: PermissionChecker,
                              val commentService: CommentService
                             ) extends Worker with CreateReleaseTaskLogic {

  override def execute: ExecuteJob = {
    case CreateReleaseTaskJob(taskRef) =>
      startTask(taskRef)
  }

  override def processResult: ProcessJobResult = {
    case result: CreateReleaseTaskExecutionResult =>
      releaseActorService.handleCreateReleaseTaskExecutionResult(result)
  }
}

trait CreateReleaseTaskLogic extends Logging {

  def eventBus: XLReleaseEventBus

  def releaseActorService: ReleaseActorService

  def releaseService: ReleaseService

  def ciIdService: CiIdService

  def authenticationService: AuthenticationService

  def permissionChecker: PermissionChecker

  def commentService: CommentService

  def startTask(taskRef: TaskSoftReference[CreateReleaseTask]): CreateReleaseTaskExecutionResult = {
    try {
      val task = taskRef.get()
      validateRequiredProperties(task)
      validateCreateReleaseRecursion(task)
      val scriptUsernamePassword = authenticationService.loginScriptUser(task)
      if (hasText(task.getCreatedReleaseId) && Ids.isReleaseId(task.getCreatedReleaseId)) {
        connectWithCreatedRelease(task)
      } else {
        createSubRelease(task, scriptUsernamePassword)
      }
    } catch {
      case ex: Exception =>
        val message = format("Exception occurred while creating a release. Reason: %s.", ex.getMessage)
        logger.error(message, ex)
        CreateReleaseTaskExecutionResulCreateReleaseFailure(taskRef.getTaskId, taskRef.getExecutionId, ex.getMessage)
    } finally {
      authenticationService.logoutScriptUser()
    }
  }

  private def connectWithCreatedRelease(task: CreateReleaseTask): CreateReleaseTaskExecutionResult = {
    val createdReleaseId = task.getCreatedReleaseId
    try {
      val maybeReleaseInfo = releaseService.getReleaseInformation(createdReleaseId).toScala
      maybeReleaseInfo match {
        case Some(releaseInfo) =>
          CreateReleaseTaskConnectReleaseSuccess(task.getId, task.getExecutionId, createdReleaseId, releaseInfo.status)
        case None =>
          val message = format("Release '%s' could not be found", createdReleaseId)
          CreateReleaseTaskConnectReleaseFailure(task.getId, task.getExecutionId, message)
      }
    } catch {
      case ex: Exception =>
        logger.error(s"Exception occurred while connecting with a release '$createdReleaseId'", ex)
        val message = format("Exception occurred while connecting with a release '%s'. Reason: '%s'", createdReleaseId, ex.getMessage)
        CreateReleaseTaskConnectReleaseFailure(task.getId, task.getExecutionId, message)
    }
  }

  private def createSubRelease(task: CreateReleaseTask, scriptUserPass: UsernamePassword) = {
    checkPermissions(task)
    val subRelease = createRelease(task, scriptUserPass)
    val startSubReleaseOnCreation = task.getStartRelease
    if (startSubReleaseOnCreation) {
      startSubRelease(task, subRelease)
    } else {
      CreateReleaseTaskExecutionResultSuccess(task.getId, task.getExecutionId, subRelease.getId)
    }
  }

  private def startSubRelease(task: CreateReleaseTask, subRelease: Release) = {
    try {
      checkPermissions(task, subRelease.getId, START_RELEASE)
      val isStarted = startRelease(task, subRelease)
      if (isStarted) {
        CreateReleaseTaskExecutionResultSuccess(task.getId, task.getExecutionId, subRelease.getId)
      } else {
        val message = format("Release '%s' could not be started", subRelease.getTitle)
        CreateReleaseTaskExecutionResultStartReleaseFailure(task.getId, task.getExecutionId, subRelease.getId, message)
      }
    } catch {
      case ex: Exception =>
        val message = format("Release '%s' could not be started. Reason: '%s'", subRelease.getTitle, ex.getMessage)
        CreateReleaseTaskExecutionResultStartReleaseFailure(task.getId, task.getExecutionId, subRelease.getId, message)
    }
  }

  private def validateCreateReleaseRecursion(task: CreateReleaseTask): Unit = {
    if (isNotBlank(task.getTemplateIdVariable) && isNotBlank(task.getRelease.getOriginTemplateId)) {
      checkArgument(task.getRelease.getOriginTemplateId != task.getTemplateId,
        "The template that will be used to create a new release must be different from the template used to create the current release.".asInstanceOf[Any])
    }
  }

  private def validateRequiredProperties(task: CreateReleaseTask): Unit = {
    checkArgument(isNotBlank(task.getNewReleaseTitle), "Title of the release to be created cannot be empty.".asInstanceOf[Any])
    val errorMessageTemplate: Any = if (isNotBlank(task.getTemplateIdVariable)) {
      "Template that will be used to create a release couldn't be derived from variable " + task.getTemplateIdVariable
    } else {
      "Template that will be used to create a release must be specified."
    }
    checkArgument(isNotBlank(task.getTemplateId), errorMessageTemplate)
    checkArgument(releaseService.exists(task.getTemplateId),
      s"Template provided in variable ${task.getTemplateIdVariable} must be a valid template id.".asInstanceOf[Any])
  }

  private def checkPermissions(task: CreateReleaseTask): Unit = {
    checkPermissions(task, task.getTemplateId, null)
  }

  private def checkPermissions(task: CreateReleaseTask, releaseId: String, requiredPermission: Permission): Unit = {
    // in case template is moved and target folder id is not specified create release task needs to create release in the new folder of the template
    val templateId = releaseService.getFullId(Ids.getFolderlessId(task.getTemplateId))
    val targetFolderId = if (!Ids.isNullId(task.getFolderId)) {
      task.getFolderId
    } else {
      Ids.findFolderId(templateId)
    }
    permissionChecker.checkIsAllowedToCreateReleaseFromTemplate(templateId, targetFolderId)
    if (requiredPermission != null) {
      permissionChecker.check(requiredPermission, releaseId)
    }
  }

  private def createRelease(task: CreateReleaseTask, scriptUsernamePass: UsernamePassword): Release = {
    val subReleaseTemplate = releaseService.findById(task.getTemplateId)
    val releaseMetadata = getSubReleaseParamsFromTask(task, subReleaseTemplate)
    val rootReleaseId = Optional.ofNullable(task.getRelease.getRootReleaseId).orElse(task.getRelease.getId)
    try throwIfMaxConcurrencyReached(releaseService.findByIdIncludingArchived(rootReleaseId))
    catch {
      case ex: NotFoundException =>
        logger.debug(format("Root release not found. Missing release max concurrency check. Reason: %s.", ex.getMessage), ex)
    }
    releaseMetadata.setRootReleaseId(rootReleaseId)
    // inherit owner, scriptUsername, scriptPassword from task's release
    releaseMetadata.setOwner(scriptUsernamePass.username)
    releaseMetadata.setScriptUsername(scriptUsernamePass.username)
    // Password should be already decrypted
    releaseMetadata.setScriptUserPassword(scriptUsernamePass.password)

    //As we check if user are passing encryptedValues or not, it is required to decrypt internally created releaseMetadata for create release task
    EncryptionHelper.decrypt(releaseMetadata)
    val subRelease = releaseService.createFromTemplate(task.getTemplateId, releaseMetadata, task.getFolderId)
    commentService.create(task, format("Created release %s.", getReleaseLink(subRelease)), AUTHENTICATED_USER, true)
    eventBus.publish(ReleaseCreatedEvent(subRelease, CreatedFromCreateReleaseTask(task, subReleaseTemplate.getId)))
    eventBus.publish(ReleaseCreatedFromCreateReleaseTaskEvent(task, subRelease))
    subRelease
  }

  private def startRelease(task: CreateReleaseTask, subRelease: Release): Boolean = {
    val startedRelease = releaseActorService.startRelease(subRelease.getId, SYSTEM)
    val hasBeenStarted = startedRelease.getStatus.hasBeenStarted
    if (hasBeenStarted) {
      commentService.create(task.getId, format("Started release %s.", getReleaseLink(startedRelease)), AUTHENTICATED_USER, true)
      eventBus.publish(ReleaseStartedFromCreateReleaseTaskEvent(task, subRelease))
    }
    hasBeenStarted
  }

  private def throwIfMaxConcurrencyReached(rootRelease: Release): Unit = {
    val maxConcurrency = rootRelease.getMaxConcurrentReleases
    val spawnedReleases = releaseService.findSpawnedReleases(rootRelease.getId, maxConcurrency)
    val rootCount = if (rootRelease.isDefunct) 0 else 1
    val currentlyRunning = spawnedReleases.size + rootCount
    logger.debug("maxConcurrency: {}, currently running releases ({}): {}", maxConcurrency, currentlyRunning, spawnedReleases)
    if (currentlyRunning >= maxConcurrency) {
      throw new IllegalStateException(String.format("Exceeded maximum number of concurrent releases created from %s. "
        + "Current maximum is %s. This value can be configured in deployit-defaults.properties: "
        + "xlrelease.Release.maxConcurrentReleases", rootRelease.getTitle, maxConcurrency))
    }
  }

  private def getSubReleaseParamsFromTask(task: CreateReleaseTask, subReleaseTemplate: Release): Release = {
    val templateVariables: JMap[String, Variable] = VariableHelper.indexByKey(subReleaseTemplate.getVariables)
    import scala.jdk.CollectionConverters._
    val variablesShownOnReleaseStart: JList[Variable] = task.getTemplateVariables.asScala.filter(
      (variable: Variable) => templateVariables.containsKey(variable.getKey) && templateVariables.get(variable.getKey).getShowOnReleaseStart
    ).asJava
    val variables: JList[Variable] = fillVariableValues(subReleaseTemplate.getVariables, variablesShownOnReleaseStart)
    val subReleaseTags = new JArrayList[String](subReleaseTemplate.getTags)
    subReleaseTags.addAll(task.getReleaseTags)
    val release: Release = newRelease
      .withVariables(variables)
      .withOriginTemplateId(task.getTemplateId)
      .withTitle(task.getNewReleaseTitle)
      .withOwner(task.getRelease.getScriptUsername)
      .withScriptUsername(task.getRelease.getScriptUsername)
      .withScriptUserPassword(task.getRelease.getScriptUserPassword)
      .withDisableNotifications(task.getRelease.isDisableNotifications)
      .withTags(subReleaseTags)
      .build
    release.setProperty(RISK_PROFILE, task.getProperty(RISK_PROFILE))
    releaseService.setDatesFromTemplate(release, subReleaseTemplate)
    release.setStartedFromTaskId(task.getId)
    release.setAbortOnFailure(subReleaseTemplate.isAbortOnFailure)
    release.setAllowPasswordsInAllFields(subReleaseTemplate.isAllowPasswordsInAllFields)
    release.setArchiveRelease(subReleaseTemplate.isArchiveRelease)
    fixUpVariableIds(release.getId, release.getVariables, ciIdService)
    release
  }

  private def getReleaseLink(release: Release) = format("[%s](#/releases/%s)", release.getTitle, DOMAIN.convertToViewId(release.getId))
}
