package com.xebialabs.xlrelease.script.el

import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.dataformat.yaml.YAMLMapper
import com.xebialabs.xlrelease.domain.Task
import com.xebialabs.xlrelease.script.el.ElPreconditionContext._

import java.util.{Map => JavaMap}
import scala.jdk.CollectionConverters._

/**
 * Secure context for EL precondition evaluation.
 * Only provides safe, read-only access to release data and utility functions.
 * All methods are designed to be safe and non-destructive.
 *
 * @param task             The current task being evaluated
 * @param releaseVariables Map of release-level variables
 * @param folderVariables  Map of folder-level variables
 * @param globalVariables  Map of global variables
 */
class ElPreconditionContext(val task: Task,
                            val releaseVariables: JavaMap[String, Object],
                            val folderVariables: JavaMap[String, Object],
                            val globalVariables: JavaMap[String, Object]) {

  /**
   * Performs case-insensitive string comparison.
   * This method safely handles null values by treating them as equal only to other null values.
   *
   * @param str1 The first string to compare (can be null)
   * @param str2 The second string to compare (can be null)
   * @return true if both strings are equal (ignoring case) or both are null, false otherwise
   * @example
   * {{{
   * equalsIgnoreCase("Test", "TEST") // returns true
   * equalsIgnoreCase("Hello", "world") // returns false
   * equalsIgnoreCase(null, null) // returns true
   * equalsIgnoreCase("test", null) // returns false
   * }}}
   */
  def equalsIgnoreCase(str1: String, str2: String): Boolean = {
    if (str1 == null && str2 == null) {
      true
    } else if (str1 == null || str2 == null) {
      false
    } else {
      str1.equalsIgnoreCase(str2)
    }
  }

  /**
   * Checks if a string contains a substring (case-insensitive).
   * This method safely handles null values by returning false if either parameter is null.
   *
   * @param str       The string to search in (can be null)
   * @param substring The substring to search for (can be null)
   * @return true if str contains substring (ignoring case), false if either parameter is null or substring is not found
   * @example
   * {{{
   * containsIgnoreCase("Production Environment", "PROD") // returns true
   * containsIgnoreCase("Hello World", "hello") // returns true
   * containsIgnoreCase("test", "xyz") // returns false
   * containsIgnoreCase(null, "test") // returns false
   * }}}
   */
  def containsIgnoreCase(str: String, substring: String): Boolean = {
    if (str == null || substring == null) {
      false
    } else {
      str.toLowerCase.contains(substring.toLowerCase)
    }
  }

  /**
   * Checks if a string starts with a specific prefix (case-sensitive).
   * This method safely handles null values by returning false if either parameter is null.
   *
   * @param str    The string to check (can be null)
   * @param prefix The prefix to check for (can be null)
   * @return true if str starts with prefix (case-sensitive), false if either parameter is null or prefix is not found
   * @example
   * {{{
   * startsWith("Production Environment", "Prod") // returns true
   * startsWith("Production Environment", "prod") // returns false (case-sensitive)
   * startsWith("Hello", "Hello World") // returns false
   * startsWith(null, "test") // returns false
   * }}}
   */
  def startsWith(str: String, prefix: String): Boolean = {
    if (str == null || prefix == null) {
      false
    } else {
      str.startsWith(prefix)
    }
  }

  /**
   * Checks if a string starts with a specific prefix (case-insensitive).
   * This method safely handles null values by returning false if either parameter is null.
   *
   * @param str    The string to check (can be null)
   * @param prefix The prefix to check for (can be null)
   * @return true if str starts with prefix (ignoring case), false if either parameter is null or prefix is not found
   * @example
   * {{{
   * startsWithIgnoreCase("Production Environment", "PROD") // returns true
   * startsWithIgnoreCase("Production Environment", "prod") // returns true
   * startsWithIgnoreCase("Hello", "HELLO WORLD") // returns false
   * startsWithIgnoreCase(null, "test") // returns false
   * }}}
   */
  def startsWithIgnoreCase(str: String, prefix: String): Boolean = {
    if (str == null || prefix == null) {
      false
    } else {
      str.toLowerCase.startsWith(prefix.toLowerCase)
    }
  }

  /**
   * Checks if a string ends with a specific suffix (case-sensitive).
   * This method safely handles null values by returning false if either parameter is null.
   *
   * @param str    The string to check (can be null)
   * @param suffix The suffix to check for (can be null)
   * @return true if str ends with suffix (case-sensitive), false if either parameter is null or suffix is not found
   * @example
   * {{{
   * endsWith("Production Environment", "ment") // returns true
   * endsWith("Production Environment", "MENT") // returns false (case-sensitive)
   * endsWith("Hello", "Hello World") // returns false
   * endsWith(null, "test") // returns false
   * }}}
   */
  def endsWith(str: String, suffix: String): Boolean = {
    if (str == null || suffix == null) {
      false
    } else {
      str.endsWith(suffix)
    }
  }

  /**
   * Checks if a string ends with a specific suffix (case-insensitive).
   * This method safely handles null values by returning false if either parameter is null.
   *
   * @param str    The string to check (can be null)
   * @param suffix The suffix to check for (can be null)
   * @return true if str ends with suffix (ignoring case), false if either parameter is null or suffix is not found
   * @example
   * {{{
   * endsWithIgnoreCase("Production Environment", "MENT") // returns true
   * endsWithIgnoreCase("Production Environment", "ment") // returns true
   * endsWithIgnoreCase("Hello", "HELLO WORLD") // returns false
   * endsWithIgnoreCase(null, "test") // returns false
   * }}}
   */
  def endsWithIgnoreCase(str: String, suffix: String): Boolean = {
    if (str == null || suffix == null) {
      false
    } else {
      str.toLowerCase.endsWith(suffix.toLowerCase)
    }
  }

  /**
   * Universal contains function that works with different data types.
   * This method supports searching in various data structures and performs case-insensitive matching.
   *
   * Behavior by search type:
   * - Collection: Returns true if item is an element in the Collection (case-insensitive string comparison)
   * - Map: Returns true if item is found in any of the Map's values (case-insensitive string comparison)
   * - Array: Returns true if item is an element in the Array (case-insensitive string comparison)
   * - String: Returns true if item is a substring of search (case-insensitive)
   * - Other types: Converts to string and performs substring search (case-insensitive)
   *
   * @param search The object to search in (Collection, Map, Array, String, or any other type)
   * @param item   The item to search for (will be converted to string for comparison)
   * @return true if item is found in search, false if either parameter is null or item is not found
   * @example
   * {{{
   * contains("Hello World", "llo") // returns true
   * contains(List("admin", "user"), "ADMIN") // returns true (case-insensitive)
   * contains(Map("key1" -> "value1"), "VALUE1") // returns true (searches values)
   * contains(Array(1, 2, 3), "2") // returns true (converts to string)
   * contains(12345, "234") // returns true (converts number to string)
   * contains(null, "test") // returns false
   * }}}
   */
  def contains(search: Any, item: Any): Boolean = {
    Option(search).zip(Option(item)).exists { case (s, i) =>
      val itemStr = String.valueOf(i).toLowerCase

      s match {
        case c: java.util.Collection[_] =>
          c.asScala.exists(e => String.valueOf(e).equalsIgnoreCase(itemStr))
        case m: java.util.Map[_, _] =>
          m.values().asScala.exists(v => String.valueOf(v).equalsIgnoreCase(itemStr))
        case a: Array[_] =>
          a.exists(e => String.valueOf(e).equalsIgnoreCase(itemStr))
        case str: String =>
          str.toLowerCase.contains(itemStr)
        case other =>
          String.valueOf(other).toLowerCase.contains(itemStr)
      }
    }
  }

  /**
   * Retrieves a variable value from the variable hierarchy.
   * Searches for the variable in the following order: release variables, folder variables, global variables.
   * Returns the first match found or null if the variable doesn't exist in any scope.
   *
   * @param name The name of the variable to retrieve
   * @return The variable value (Object) or null if not found
   * @example
   * {{{
   * getVariable("environment") // returns "production" if variable exists
   * getVariable("nonexistent") // returns null
   * }}}
   */
  def getVariable(name: String): Object = {
    Option(releaseVariables.get(name))
      .orElse(Option(folderVariables.get(name)))
      .orElse(Option(globalVariables.get(name)))
      .orNull
  }

  /**
   * Checks if a variable exists in any of the variable scopes.
   * Searches for the variable in release variables, folder variables, and global variables.
   *
   * @param name The name of the variable to check for existence
   * @return true if the variable exists in any scope (release, folder, or global), false otherwise
   * @example
   * {{{
   * hasVariable("environment") // returns true if variable exists in any scope
   * hasVariable("nonexistent") // returns false
   * }}}
   */
  def hasVariable(name: String): Boolean = {
    releaseVariables.containsKey(name) ||
      folderVariables.containsKey(name) ||
      globalVariables.containsKey(name)
  }

  /**
   * Returns a set containing all available variable names from all scopes.
   * This method combines variable names from release, folder, and global variable scopes.
   * Useful for debugging and inspection purposes.
   *
   * @return A Java Set containing all unique variable names across all scopes
   * @example
   * {{{
   * getAvailableVariables() // returns Set("var1", "var2", "global.setting", ...)
   * getAvailableVariables().size() // returns total number of unique variables
   * }}}
   */
  def getAvailableVariables(): java.util.Set[String] = {
    val allKeys = scala.collection.mutable.Set[String]()
    allKeys ++= releaseVariables.keySet().asScala
    allKeys ++= folderVariables.keySet().asScala
    allKeys ++= globalVariables.keySet().asScala
    allKeys.asJava
  }

  /**
   * Parses a JSON string into a Java Map.
   * This method provides safe JSON parsing functionality for use in expressions.
   * The returned map can be used to access nested JSON properties.
   *
   * @param input The JSON string to parse
   * @return A Java Map representing the parsed JSON structure
   * @throws com.fasterxml.jackson.core.JsonProcessingException if the input is not valid JSON
   * @example
   * {{{
   * parseJson('{"name": "John", "age": 30}')["name"] // returns "John"
   * parseJson('{"user": {"roles": ["admin"]}}')["user"]["roles"] // returns ["admin"]
   * }}}
   */
  def parseJson(input: String): JavaMap[String, Object] = {
    objectMapper.readValue(input, classOf[java.util.Map[String, Object]])
  }

  /**
   * Parses a YAML string into a Java Map.
   * This method provides safe YAML parsing functionality for use in expressions.
   * The returned map can be used to access nested YAML properties.
   *
   * @param input The YAML string to parse
   * @return A Java Map representing the parsed YAML structure
   * @throws com.fasterxml.jackson.core.JsonProcessingException if the input is not valid YAML
   * @example
   * {{{
   * parseYaml('name: John\nage: 30')["name"] // returns "John"
   * parseYaml('user:\n  roles:\n    - admin')["user"]["roles"] // returns ["admin"]
   * }}}
   */
  def parseYaml(input: String): JavaMap[String, Object] = {
    yamlMapper.readValue(input, classOf[java.util.Map[String, Object]])
  }
}

/**
 * Companion object for ElPreconditionContext.
 * Contains shared instances of JSON and YAML mappers used for parsing operations.
 * These mappers are lazily initialized to improve performance and memory usage.
 */
object ElPreconditionContext {
  /** Lazy-initialized ObjectMapper for JSON parsing operations */
  private lazy val objectMapper = new ObjectMapper()

  /** Lazy-initialized YAMLMapper for YAML parsing operations */
  private lazy val yamlMapper = new YAMLMapper()
}
