package com.xebialabs.xlrelease.script.el

import com.xebialabs.xlrelease.domain.{PreconditionType, Task}
import com.xebialabs.xlrelease.script._
import com.xebialabs.xlrelease.security.authentication.AuthenticationService
import com.xebialabs.xlrelease.service.CommentService
import grizzled.slf4j.Logging
import org.springframework.expression.{EvaluationContext, ExpressionParser}
import org.springframework.security.core.context.SecurityContextHolder
import org.springframework.util.StringUtils

class ElPreconditionService(authenticationService: AuthenticationService,
                            commentService: CommentService,
                            parser: ExpressionParser,
                            evaluationContextBuilder: EvaluationContextBuilder
                           )
  extends PreconditionService with Logging {

  override def engine(): String = PreconditionType.EL.engine

  override def executePrecondition(task: Task): PreconditionResult = {
    val result = try {
      authenticationService.loginScriptUser(task)
      executeElPrecondition(task)
    } finally {
      authenticationService.logoutScriptUser()
    }
    if (StringUtils.hasText(result.executionLog)) {
      commentService.appendComment(task, null, result.executionLog)
    }
    result
  }

  private def executeElPrecondition(task: Task): PreconditionResult = {
    val currentAuthentication = SecurityContextHolder.getContext.getAuthentication
    val result = try {
      val expression = parser.parseExpression(task.getPrecondition)
      val context: EvaluationContext = evaluationContextBuilder.buildPreconditionContext(task)
      val expressionResult = expression.getValue(context, classOf[Boolean])
      if (expressionResult) {
        ValidPreconditionResult(task.getId, task.getExecutionId, "Precondition is valid (returned True)", None, currentAuthentication)
      } else {
        InvalidPreconditionResult(task.getId, task.getExecutionId, "Precondition is invalid (returned a value that is not True)", None, currentAuthentication)
      }
    } catch {
      case exception: Exception =>
        logger.debug("Precondition evaluation failed.", exception)
        ExceptionPreconditionResult(
          task.getId,
          task.getExecutionId,
          f"Precondition evaluation failed. Failure: ${exception.getMessage}",
          None,
          currentAuthentication
        )
    }
    result
  }

}
