package com.xebialabs.xlrelease.service

import com.xebialabs.xlrelease.actors.initializer.{ManagedActorInitializer, ManagedActorInitializerState}
import com.xebialabs.xlrelease.domain.management.{CompositeServiceStatus, ServiceState, ServiceStatus, SimpleServiceStatus}
import org.springframework.stereotype.Component

@Component
class ActorsLifecycleManager(managedInitializers: Array[ManagedActorInitializer])
  extends XlrServiceLifecycle {

  override def getOrder: Int = XlrServiceLifecycleOrder.ACTORS

  override def doStart(): Unit = {
    startInitializers()
  }

  override def doStop(): Unit = {
    stopInitializers()
  }

  private def adaptToDomainState(initializerState: ManagedActorInitializerState.State): ServiceState.ServiceState = {
    initializerState match {
      case ManagedActorInitializerState.Running => ServiceState.Running
      case ManagedActorInitializerState.Stopping => ServiceState.Stopping
      case ManagedActorInitializerState.Stopped => ServiceState.Stopped
    }
  }

  override def serviceStatus(): ServiceStatus = {
    val allManagedActorInitializerStatuses: List[ServiceStatus] = managedInitializers.map { initializer =>
      SimpleServiceStatus(initializer.serviceName(), adaptToDomainState(initializer.getStatus()))
    }.toList
    CompositeServiceStatus(serviceName(), state, allManagedActorInitializerStatuses)
  }

  private def startInitializers(): Unit = {
    managedInitializers.foreach { initializer =>
      try {
        initializer.start()
      } catch {
        case e: Exception =>
          throw new IllegalStateException(s"Failed to start initializer ${initializer.serviceName()}", e)
      }
    }
  }

  private def stopInitializers(): Unit = {
    managedInitializers.reverse.foreach { initializer =>
      try {
        initializer.stop()
      } catch {
        case e: Exception =>
          throw new IllegalStateException(s"Failed to stop initializer ${initializer.serviceName()}", e)
      }
    }
  }
}
