package com.xebialabs.xlrelease.service

import com.xebialabs.xlplatform.cluster.ClusterMode._
import com.xebialabs.xlrelease.actors.BroadcastEventActor.PublishEvent
import com.xebialabs.xlrelease.actors.{ActorSystemHolder, BroadcastEventActor}
import com.xebialabs.xlrelease.domain.distributed.events.DistributedXLReleaseEvent
import com.xebialabs.xlrelease.events.EventBus
import org.springframework.context.annotation.Profile
import org.springframework.stereotype.Component

trait BroadcastService {
  def broadcast(event: DistributedXLReleaseEvent, publishEventOnSelf: Boolean): Unit
}

@Component
@Profile(Array(FULL))
class ClusterBroadcastService(actorSystemHolder: ActorSystemHolder) extends BroadcastService {

  private lazy val broadcastEventActor = actorSystemHolder.actorOf(classOf[BroadcastEventActor], "broadcastEventActor")

  // Be careful when broadcasting events from initializers and upgraders. Broadcast service will not be available until actor system starts
  // Currently only cache evict events are present on first boot, ideally we do no add any new behavior there.
  override def broadcast(event: DistributedXLReleaseEvent, publishEventOnSelf: Boolean): Unit = {
    if (actorSystemHolder.isActorSystemInitialized && broadcastEventActor.isInitialized) {
      broadcastEventActor ! PublishEvent(event, publishEventOnSelf)
    }
  }
}

@Component
@Profile(Array(STANDALONE, HOT_STANDBY))
class NonClusteredBroadcastService(eventBus: EventBus) extends BroadcastService {

  override def broadcast(event: DistributedXLReleaseEvent, publishEventOnSelf: Boolean): Unit = {
    if (publishEventOnSelf) eventBus.publish(event)
  }

}
