package com.xebialabs.xlrelease.service

import com.google.common.annotations.VisibleForTesting
import com.google.common.base.Preconditions.checkState
import com.xebialabs.xlrelease.domain.{Phase, Release, Task}
import com.xebialabs.xlrelease.repository.{Ids, PhaseVersion}
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.service.PhaseRestart.{RESTART_PHASE_ERROR, RestartPhaseResult}
import com.xebialabs.xlrelease.utils.CiHelper.TO_ID
import io.micrometer.core.annotation.Timed

import java.util.{List => JList}
import scala.jdk.CollectionConverters._

trait PhaseRestart {

  /**
   * Restart phases from the given release, phase and (optional) task id.
   *
   * @param release
   * @param phaseId
   * @param taskId
   * @return Release with the restarted phases
   */
  @Timed
  def restartPhases(releaseId: String, phaseId: String, taskId: String, phaseVersion: PhaseVersion, release: Release): RestartPhaseResult =
    restartPhases(releaseId, phaseId, taskId, phaseVersion, resumeRelease = false, release)

  @Timed
  def restartPhases(releaseId: String, phaseId: String, taskId: String, phaseVersion: PhaseVersion, resumeRelease: Boolean, release: Release): RestartPhaseResult

  @Timed
  def restorePhases(phasesToRestore: JList[Phase]): JList[Phase]

  def checkCanRestartPhases(release: Release, phaseId: String, taskId: String, phaseVersion: PhaseVersion, resumeRelease: Boolean): Unit = {
    if (!release.hasNoAutomatedTaskRunning) {
      throw new RestartPhasesException(release.getId, phaseId, taskId, phaseVersion, resumeRelease, RESTART_PHASE_ERROR)
    }
    if (resumeRelease) {
      release.setAutomatedResumeCount(release.getAutomatedResumeCount + 1)
      if (release.getAutomatedResumeCount > release.getMaxAutomatedResumes) {
        throw new RestartPhasesException(release.getId, phaseId, taskId, phaseVersion, resumeRelease,
          s"Exceeded maximum number [${release.getMaxAutomatedResumes}] of automated release resumes." +
            s" Please resume the release manually or update the 'xlrelease.Release.maxAutomatedResumes' property in the deployit-defaults.properties file.")

      }
    }
  }

  def updateGatesReferencingPhases(release: Release, phasesToRestoreIds: Seq[String], restoredPhasesIds: Seq[String]): Unit
}

object PhaseRestart {
  val RESTART_PHASE_ERROR = "You attempted to restart phases on a release with running automated tasks"

  case class RestartPhaseResult(release: Release, phaseIdsToRestore: Seq[String], restoredPhaseIds: Seq[String])

  @VisibleForTesting
  private[service] def getPhasesToRestore(release: Release, phaseId: String, phaseVersion: PhaseVersion): JList[Phase] = {
    require(release.hasPhase(phaseId), s"Phase with id $phaseId not found in release '${release.getTitle}'.")
    require(release.hasCurrentPhase, s"Release '${release.getTitle}' must have a current phase in order to restart phases")

    val (from, until) = getPhasesRangeToRestore(release, phaseId, phaseVersion)
    require(from <= until, "Can't restore a phase that is after the current phase.")

    val phasesToRestore = release.getPhases.asScala.slice(from, until + 1)
    phaseVersion match {
      case PhaseVersion.LATEST =>
        phasesToRestore
          .filter(p =>
            !p.isPhaseCopied && p.isLatestCopy
          )
      case PhaseVersion.ORIGINAL =>
        phasesToRestore.filter(_.isOriginal)
      case _ =>
        // do nothing, old behavior copies all between given and current phase
        phasesToRestore
    }
  }.asJava

  @VisibleForTesting
  private[service] def getPhasesRangeToRestore(release: Release, phaseId: String, phaseVersion: PhaseVersion): (Int, Int) = {
    val firstPhaseToRestore = release.getPhase(phaseId)
    val from = release.getPhases.indexOf(firstPhaseToRestore)
    val currentPhase = release.getCurrentPhase
    var until = release.getPhases.indexOf(currentPhase)

    if (phaseVersion == PhaseVersion.ORIGINAL && !currentPhase.isOriginal) {
      until = from
      var phase = firstPhaseToRestore
      while (phase.getId != currentPhase.getOriginId) {
        phase = release.getNextPhase(phase)
        until = until + 1
      }
    }
    (from, until)
  }

  def checkIfPhaseCanBeRestartedFrom(task: Task, phase: Phase, permissions: PermissionChecker): Unit = {
    if (task != null) {
      checkState(task.isDone || task.isActive, "You can not restart from a task that hasn't been started", null)

      val tasks = phase.getAllTasks.asScala

      checkState(!isTaskInParallelGroup(task.getId, tasks.asJava), "You can not restart from a task in a parallel group", task.getId)

      val taskIds = tasks.map(TO_ID(_))
      val taskIndexToRestartFrom = taskIds.indexOf(task.getId)

      tasks.take(taskIndexToRestartFrom).foreach(permissions.checkHasPermissionsToUpdateTask)
    }
  }

  private def isTaskInParallelGroup(taskId: String, tasks: JList[Task]): Boolean = {
    val taskMap: Map[String, Task] = tasks.asScala.map(task => task.getId -> task).toMap

    @scala.annotation.tailrec
    def checkParent(parent: String): Boolean = {
      if (parent == null || !Ids.isTaskId(parent)) {
        false
      } else {
        taskMap.get(parent) match {
          case Some(parentTask) if parentTask.isParallelGroup => true
          case _ => checkParent(Ids.getParentId(parent))
        }
      }
    }

    checkParent(Ids.getParentId(taskId))
  }
}
