package com.xebialabs.xlrelease.service

import com.xebialabs.xlplatform.sugar.PathSugar.path2File
import com.xebialabs.xlrelease.domain.TemplateLogo
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException
import com.xebialabs.xlrelease.repository.TemplateMetadataRepository
import grizzled.slf4j.Logging
import org.apache.commons.io.IOUtils
import org.springframework.boot.context.event.ApplicationReadyEvent
import org.springframework.context.event.EventListener
import org.springframework.stereotype.Service
import org.springframework.util.FileSystemUtils

import java.io.{File, FileOutputStream}
import java.nio.file.{Path, Paths}
import scala.util.Using

@Service
class TemplateMetadataService(templateMetadataRepository: TemplateMetadataRepository)
  extends Logging with WorkdirSupport {

  private val cacheDir: Path = Paths.get("cache")
  private val logoDir = cacheDir.resolve(TemplateLogo.EXPORT_DIRECTORY)

  def getLogo(logoId: String): File = this.synchronized {
    makeDirs(logoDir)

    // Can be optimised if we get actually filename
    val templateLogo = templateMetadataRepository.findTemplateLogoById(logoId)
      .getOrElse(throw new LogFriendlyNotFoundException("Repository entity [%s] not found", logoId))
    val maybeLogoFile = new File(cacheDir, templateLogo.getExportFilename)

    if (maybeLogoFile.exists() && maybeLogoFile.isFile) {
      maybeLogoFile
    } else {
      fetchLogoFromDatabase(templateLogo)
    }
  }

  private def fetchLogoFromDatabase(templateLogo: TemplateLogo): File = {
    val result = withWorkDir {
      Using.Manager { use =>
        val newFile = new File(cacheDir, templateLogo.getExportFilename)
        val fos = use(new FileOutputStream(newFile))
        val is = use(templateLogo.getFile.getInputStream)
        IOUtils.copy(is, fos)
        newFile
      }
    }
    result.get
  }

  private def makeDirs(dirPath: Path): Unit = {
    if (!dirPath.toFile.isDirectory) {
      dirPath.mkdirs()
    }
  }

  private def cleanCache(): Unit = {
    try {
      FileSystemUtils.deleteRecursively(logoDir)
    } catch {
      case e => logger.warn(s"Unable to delete $logoDir")
    }
  }

  @EventListener
  def onApplicationReadyEvent(event: ApplicationReadyEvent): Unit = {
    cleanCache()
  }
}