package com.xebialabs.xlrelease.service

import com.xebialabs.xlplatform.cluster.full.downing.LeaderAutoDowningActor.NODE_DOWNED_EXIT_CODE
import com.xebialabs.xlrelease.domain.events.{ClusterNodeStartEvent, ClusterNodeStopEvent}
import com.xebialabs.xlrelease.domain.management.{ServiceState, ServiceStatus}
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.support.pekko.spring.ScalaSpringAwareBean
import grizzled.slf4j.Logging
import org.springframework.boot.SpringApplication
import org.springframework.boot.actuate.health.Status
import org.springframework.security.core.context.{SecurityContext, SecurityContextHolder}
import org.springframework.stereotype.Service

import scala.concurrent.{ExecutionContextExecutor, Future}
import scala.util.{Failure, Success}

@Service
class XlrServiceManager(managers: Array[XlrServiceLifecycle], eventBus: XLReleaseEventBus) extends Logging with ScalaSpringAwareBean {

  def startAsync(): Unit = {
    val securityContext: SecurityContext = SecurityContextHolder.getContext
    implicit val ec: ExecutionContextExecutor = scala.concurrent.ExecutionContext.global
    Future {
      SecurityContextHolder.setContext(securityContext)
      try {
        start()
      } finally {
        SecurityContextHolder.clearContext()
      }
    } onComplete {
      case Success(true) => logger.info("All services started")
      case Success(false) => logger.warn("Some services are already running")
      case Failure(e) => {
        logger.error("Failed to start services", e)
        SpringApplication.exit(applicationContext, () => NODE_DOWNED_EXIT_CODE)
      }
    }
  }

  private def start(): Boolean = {
    eventBus.publish(ClusterNodeStartEvent())
    val serviceStatuses = this.getServicesStatus
    val isStopped = serviceStatuses.forall(_.state == ServiceState.Stopped)
    if (isStopped) {
      managers.foreach(service => {
        logger.info(s"Starting ${service.serviceName()}")
        service.start()
        logger.info(s"Started ${service.serviceName()}")
      })
      true
    } else {
      false
    }
  }

  def stopAsync(): Unit = {
    implicit val ec: ExecutionContextExecutor = scala.concurrent.ExecutionContext.global
    Future {
      stop()
    } onComplete {
      case Success(_) => logger.info("All services stopped")
      case Failure(e) => {
        logger.error("Failed to stop services", e)
        SpringApplication.exit(applicationContext, () => NODE_DOWNED_EXIT_CODE)
      }
    }
  }

  def stop(): Unit = {
    eventBus.publish(ClusterNodeStopEvent())
    managers.reverse.foreach(service => {
      logger.info(s"Stopping ${service.serviceName()}")
      service.stop()
      logger.info(s"Stopped ${service.serviceName()}")
    })
  }

  def getServicesStatus: List[ServiceStatus] = {
    managers.map(s => s.serviceStatus()).toList
  }

  def getHealthStatus: Status = {
    if (managers
      .map(s => s.serviceStatus())
      .forall(_.state == ServiceState.Running)) {
      Status.UP
    } else {
      Status.OUT_OF_SERVICE
    }
  }

}
