package com.xebialabs.xlrelease.support.report

import com.xebialabs.deployit.{ReleaseInfo, ServerConfiguration}
import com.xebialabs.license.service.LicenseService
import com.xebialabs.xlplatform.cluster.NodeState
import com.xebialabs.xlplatform.support.report.ReportDataProvider
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.support.report.repository.DataStatisticsRepository
import org.joda.time.Duration
import org.joda.time.format.PeriodFormatterBuilder
import org.springframework.beans.factory.annotation.{Autowired, Value}
import org.springframework.stereotype.Component
import org.springframework.util.StringUtils

import java.lang.management.ManagementFactory
import java.net.InetAddress
import scala.beans.BeanProperty
import scala.jdk.CollectionConverters._
import scala.util.Try

@Component
class GeneralDataProvider @Autowired()(val repository: DataStatisticsRepository,
                                       val configurationRepository: ConfigurationRepository,
                                       val serverConfiguration: ServerConfiguration,
                                       val licenseService: LicenseService) extends ReportDataProvider {

  @Value("${xl.features.analytics.blacklist}")
  var analyticsBlacklist: java.util.List[String] = _

  val name: String = "general"

  override def collectStatistics: Map[String, Any] = {
    val statistics = new GeneralUsageStatistics()
    val releaseInfo = ReleaseInfo.getReleaseInfo
    statistics.setVersion(releaseInfo.getVersion)
    statistics.setVersionDate(releaseInfo.getDate)
    val headerName = configurationRepository.getThemeSettings().getHeaderName
    statistics.setInstanceName(if (StringUtils.hasText(headerName)) headerName else "Not defined")
    statistics.setHostname(InetAddress.getLocalHost.getHostName)
    statistics.setServerUrl(serverConfiguration.getServerUrl)
    statistics.setNoHitDBQueryDuration(repository.timeDbQueryNoHit())
    statistics.setHitDBQueryDuration(repository.timeDbQueryHit())
    statistics.setDatabaseImplementation(repository.findMainDbImplementation())
    statistics.setArchiveDatabaseImplementation(repository.findArchiveDbImplementation())

    val osMbean = ManagementFactory.getOperatingSystemMXBean
    val runTimeBean = ManagementFactory.getRuntimeMXBean
    statistics.setVmArguments(maskAnalyticsBlacklistedValues(runTimeBean.getInputArguments.asScala.toList))
    statistics.setVmName(s"${runTimeBean.getVmName} ${runTimeBean.getVmVendor} ${runTimeBean.getVmVersion} (${runTimeBean.getSpecVersion})")
    statistics.setSystemName(runTimeBean.getName)
    statistics.setUpTime(formatTimeInMS(runTimeBean.getUptime))
    statistics.setOsName(osMbean.getName)
    statistics.setOsArchitecture(osMbean.getArch)
    statistics.setOsAvailableProcessors(osMbean.getAvailableProcessors)
    val runtime = Runtime.getRuntime
    Try(System.gc()).recover { case exception => logger.warn("Unable to GC due to error", exception) }
    statistics.setUsedMemory(runtime.totalMemory() - runtime.freeMemory())
    statistics.setMaxMemory(runtime.maxMemory())
    statistics.setClusterNodeActive(NodeState.isActive)
    statistics.setAnalyticsIdentifier(repository.findAnalyticsIdentifier())


    Map[String, Any]("general" -> statistics)
  }

  def formatTimeInMS(upTimeMS: Long): String = {
    val periodFormatter = new PeriodFormatterBuilder()
      .appendDays()
      .appendSuffix("d")
      .appendHours()
      .appendSuffix("h")
      .appendMinutes()
      .appendSuffix("m")
      .appendSeconds()
      .appendSuffix("s")
      .toFormatter()
    periodFormatter.print(new Duration(upTimeMS).toPeriod())
  }

  private def maskAnalyticsBlacklistedValues(vmArgsList: List[String]): String = {
    val blacklist = analyticsBlacklist.asScala.toList.map(_.toLowerCase)
    val maskedPairs = vmArgsList.map { s =>
      s.split("=", 2) match {
        case Array(key, value) if blacklist.exists(key.toLowerCase.contains) => s"$key=*****"
        case _ => s
      }
    }
    maskedPairs.mkString(",")
  }
}

class GeneralUsageStatistics {
  @BeanProperty
  var version: String = _

  @BeanProperty
  var versionDate: String = _

  @BeanProperty
  var instanceName: String = _

  @BeanProperty
  var hostname: String = _

  @BeanProperty
  var serverUrl: String = _

  @BeanProperty
  var vmArguments: String = _

  @BeanProperty
  var vmName: String = _

  @BeanProperty
  var systemName: String = _

  @BeanProperty
  var upTime: String = _

  @BeanProperty
  var osName: String = _

  @BeanProperty
  var osArchitecture: String = _

  @BeanProperty
  var osAvailableProcessors: Int = 0

  @BeanProperty
  var maxMemory: Long = 0

  @BeanProperty
  var usedMemory: Long = 0

  @BeanProperty
  var databaseImplementation: String = _

  @BeanProperty
  var archiveDatabaseImplementation: String = _

  @BeanProperty
  var noHitDBQueryDuration: Long = 0

  @BeanProperty
  var hitDBQueryDuration: Long = 0

  @BeanProperty
  var clusterNodeActive: Boolean = false

  @BeanProperty
  var analyticsIdentifier: String = _
}
