package com.xebialabs.xlrelease.support.report

import com.xebialabs.license.service.LicenseService
import com.xebialabs.xlplatform.support.report.ReportDataProvider
import com.xebialabs.xlrelease.support.report.repository.DataStatisticsRepository
import com.xebialabs.xlrelease.views.LicenseView
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.beans.BeanProperty
import scala.math.BigDecimal.RoundingMode
import scala.util.Try

@Component
class LicenseDataProvider @Autowired()(val licenseService: LicenseService,
                                       val repository: DataStatisticsRepository) extends ReportDataProvider {
  override val name: String = "license"

  override def collectStatistics: Map[String, Any] = {
    val statistics = new LicenseUsageStatistics()

    val licenseView = new LicenseView(licenseService.getLicense)
    statistics.setLicenseVersion(licenseView.getVersion)
    statistics.setLicensedProduct(licenseView.getProduct)
    statistics.setLicensedEdition(licenseView.getEdition)
    statistics.setLicensedTo(licenseView.getLicensedTo)
    statistics.setLicenseExpiresAfter(licenseView.getExpiresAfter)
    statistics.setLicenseRemainingDays(licenseService.getValidPeriod.getDays)
    statistics.setLicensedMaxUsers(Try(Option(licenseView.getMaxNumberOfUsers).map(_.toInt).getOrElse(-1)).getOrElse(-1))
    statistics.setLicenseCurrentUsers(repository.countLicensedUsers())
    statistics.setLicenseUtilizationRate(calculateUtilizationRate(statistics.getLicensedMaxUsers, statistics.getLicenseCurrentUsers))
    statistics.setLicensedRepositoryId(licenseView.getRepositoryId)

    Map[String, Any]("license" -> statistics)
  }

  private def calculateUtilizationRate(maxUsers: Int, currentUsers: Int): Float = {
    if (maxUsers <= 0 || currentUsers < 0) {
      0F
    } else {
      (BigDecimal(currentUsers) / BigDecimal(maxUsers)).setScale(2, RoundingMode.HALF_UP).floatValue
    }
  }
}

class LicenseUsageStatistics {
  @BeanProperty
  var licenseVersion: String = _

  @BeanProperty
  var licensedProduct: String = _

  @BeanProperty
  var licensedEdition: String = _

  @BeanProperty
  var licensedTo: String = _

  @BeanProperty
  var licenseExpiresAfter: String = _

  @BeanProperty
  var licenseRemainingDays: Int = 0

  @BeanProperty
  var licensedRepositoryId: String = _

  @BeanProperty
  var licensedMaxUsers: Int = 0

  @BeanProperty
  var licenseCurrentUsers: Int = 0

  @BeanProperty
  var licenseUtilizationRate: Float = 0

}
