package com.xebialabs.xlrelease.variable

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.domain.variables._
import com.xebialabs.xlrelease.domain.{Release, VisitableItem}
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.service.CiIdService

import java.util.{List => JList}
import scala.collection.mutable

object VariablePersistenceHelper {

  def scanAndBuildNewVariables(release: Release, updatedItem: VisitableItem, ciIdService: CiIdService): JList[Variable] = {
    val newVariables = release.scanAndAddNewVariables(updatedItem)

    fixUpVariableIds(release.getId, release.getVariables, ciIdService)

    newVariables
  }

  def fixUpVariableIds(parentId: String, variables: JList[Variable], ciIdService: CiIdService): Unit = {
    val ids = mutable.SortedSet[String]()

    variables.forEach { v =>
      if (v.getId == null || v.getId.isEmpty) {
        // getUniqueId checks uniqueness against the db, so it only works after all items were persisted, which isn't the case here (REL-3580)
        var id = ciIdService.getUniqueId(Type.valueOf(classOf[Variable]), parentId)
        while (ids.contains(id)) {
          id = ciIdService.getUniqueId(Type.valueOf(classOf[Variable]), parentId)
        }
        ids.add(id)
        v.setId(id)
      } else {
        v.setId(parentId + "/" + Ids.getName(v.getId))
      }
      fixUpValueProviderConfigurationId(v)
      fixUpExternalVariableIds(v)
    }
  }

  def fixUpValueProviderConfigurationId(v: Variable): Unit = {
    val provider = v.getValueProvider
    if (null != provider && (provider.getId.isNullId || provider.getId.isNotValueProviderId || provider.getId.containsNull)) {
      provider.setId(v.getId + "/" + ValueProviderConfiguration.CONFIGURATION_ID)
    }
  }

  def fixUpExternalVariableIds(v: Variable): Unit = {
    if (!v.isInstanceOf[PasswordStringVariable]) {
      return
    }

    val passwordStringVariable = v.asInstanceOf[PasswordStringVariable]

    val externalVariableValue = passwordStringVariable.getExternalVariableValue
    if (externalVariableValue == null) {
      return
    }

    if (externalVariableValue.getId.isNullId || externalVariableValue.getId.isNotValueProviderId || externalVariableValue.getId.containsNull) {
      externalVariableValue.setId(v.getId + "/" + ExternalVariableValue.CONFIGURATION_ID)
    }
  }

  def fixUpGlobalVariable(v: Variable, ciIdService: CiIdService): Unit = {
    Option(v.getId) match {
      case Some(_) =>
      case None => v.setId(ciIdService.getUniqueId(Type.valueOf(classOf[Variable]), GlobalVariables.GLOBAL_VARIABLES_ID))
    }
    fixUpVariableState(v, Some("global"), false)
  }

  def fixUpFolderVariable(v: Variable, ciIdService: CiIdService): Unit = {
    Option(v.getId) match {
      case Some(_) =>
      case None => v.setId(Ids.getName(ciIdService.getUniqueId(Type.valueOf(classOf[Variable]), "")))
    }
    fixUpVariableState(v, Some("folder"), false)
  }

  def fixUpReleaseVariable(v: Variable, releaseId: String, ciIdService: CiIdService): Unit = {
    Option(v.getId) match {
      case Some(_) =>
      case None => v.setId(ciIdService.getUniqueId(Type.valueOf(classOf[Variable]), releaseId))
    }
    fixUpVariableState(v, None, true)
  }

  def fixUpPrefix(v: Variable, prefix: String): Unit = {
    v.setKey(v.getKey.replaceFirst(s"(?i)$prefix", prefix))
  }

  private def fixUpVariableState(v: Variable, maybePrefix: Option[String], releaseFields: Boolean): Unit = {
    v.setFolderId(null) // even folder variables don't keep this in state

    maybePrefix match {
      case Some(prefix) => fixUpPrefix(v, prefix)
      case None =>
    }

    if (!releaseFields) {
      v.setShowOnReleaseStart(false)
      v.setRequiresValue(false)
    }

    Option(v.getValueProvider).foreach(provider => {
      provider.setVariable(v)
      provider.setId(s"${v.getId}/${ValueProviderConfiguration.CONFIGURATION_ID}")
    })

    fixUpExternalVariableIds(v)
  }

  private implicit class IdExtensions(val id: String) extends AnyVal {
    def containsNull: Boolean = {
      id != null && id.contains("null")
    }

    def isNotValueProviderId: Boolean = {
      !isValueProviderId
    }

    def isValueProviderId: Boolean = {
      id != null && Ids.isValueProviderId(id)
    }

    def isNullId: Boolean = {
      id != null && Ids.isNullId(id)
    }
  }

  def checkSameType(variable: Variable, existing: Variable): Unit = {
    if (!variable.getType.equals(existing.getType)) {
      throw new IllegalArgumentException(s"Cannot change type of variable ${existing.getId} from ${variable.getType} to ${existing.getType}")
    }
  }
}
