package com.xebialabs.xlrelease.actors.cluster

import com.typesafe.config.{Config, ConfigFactory}
import com.xebialabs.xlplatform.cluster.ClusterMode.{FULL, Full, HotStandby}
import com.xebialabs.xlplatform.cluster.XlCluster
import com.xebialabs.xlplatform.cluster.full.downing.LeaderAutoDowningActor.NODE_DOWNED_EXIT_CODE
import com.xebialabs.xlrelease.actors.ActorSystemHolder
import com.xebialabs.xlrelease.actors.cluster.ClusterEventsListenerActor.Register
import com.xebialabs.xlrelease.actors.cluster.XlrCluster.XL_DOWNING_PROVIDERS
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.service.{XlrServiceLifecycle, XlrServiceLifecycleOrder}
import com.xebialabs.xlrelease.ssl.{SslContextConfig, SslContextFactory}
import com.xebialabs.xlrelease.support.config.TypesafeConfigExt.ExtendedConfig
import com.xebialabs.xlrelease.support.pekko.spring.ScalaSpringAwareBean
import grizzled.slf4j.Logging
import org.apache.pekko.actor.{ActorSystem, Address}
import org.apache.pekko.http.scaladsl.{ConnectionContext, HttpsConnectionContext}
import org.apache.pekko.management.cluster.bootstrap.ClusterBootstrap
import org.apache.pekko.management.scaladsl.PekkoManagement
import org.springframework.boot.SpringApplication

import java.util.concurrent.{ScheduledExecutorService, TimeUnit}
import scala.concurrent.duration.Duration
import scala.concurrent.{Await, ExecutionContextExecutor, Promise}
import scala.util.{Failure, Success}

trait XlrActorSystemBootstrap extends Logging with XlrServiceLifecycle {

  override def getOrder(): Int = XlrServiceLifecycleOrder.ACTOR_SYSTEM

  override def serviceName(): String = "ActorSystemLifecycleManager"

  def auxiliaryExecutor: ScheduledExecutorService

  def actorSystemHolder: ActorSystemHolder

  def boot(): Unit

  override def awaitStart(): Unit = {
    val cluster = actorSystemHolder.cluster()
    val promise = Promise[Address]()

    def checkLeader(): Unit = {
      val maybeLeader = cluster.state.leader
      if (maybeLeader.isDefined) {
        promise.success(maybeLeader.get)
      } else {
        auxiliaryExecutor.schedule(
          new Runnable {
            override def run(): Unit = checkLeader()
          },
          XlrServiceLifecycle.AWAIT_CHECK_INTERVAL,
          TimeUnit.MILLISECONDS
        )
      }
    }

    checkLeader()

    val waitingForStart: Address = Await.result(promise.future, Duration.Inf)
    logger.info(s"Cluster started, leader is $waitingForStart")
  }

  override def doStart(): Unit = {
    boot()
  }

  def doStop(): Unit = {
    actorSystemHolder.stop()
  }
}


case class XlrLegacyClusterBootstrap(xlrConfig: XlrConfig,
                                     actorSystemHolder: ActorSystemHolder,
                                     auxiliaryExecutor: ScheduledExecutorService
                                    ) extends XlrActorSystemBootstrap {


  override def boot(): Unit = {
    xlrConfig.cluster.mode match {
      case HotStandby =>
        throw new UnsupportedOperationException(
          s"HotStandby cluster mode is no longer supported, switch to $FULL if you want to boot in the cluster mode.")
      case Full =>
        val mgmtActorSystem = actorSystemHolder.createActorSystem()
        XlCluster.init(xlrConfig.cluster)
        XlCluster(mgmtActorSystem).start(() => {})
      case _ => throw new UnsupportedOperationException(s"Unsupported cluster mode \"${xlrConfig.cluster.mode}\" for legacy cluster manager")
    }
  }
}

case class XlrPekkoNativeClusterBootstrap(xlrConfig: XlrConfig,
                                          actorSystemHolder: ActorSystemHolder,
                                          auxiliaryExecutor: ScheduledExecutorService
                                         ) extends XlrActorSystemBootstrap with Logging with ScalaSpringAwareBean {

  override def boot(): Unit = {
    xlrConfig.cluster.mode match {
      case Full =>
        val system = actorSystemHolder.createActorSystem()
        implicit val ec: ExecutionContextExecutor = system.dispatcher
        val management = PekkoManagement(system)
        val managementFuture = if (useSsl) {
          val httpsServer: HttpsConnectionContext = getHttpsConnectionContext()
          management.start(_.withHttpsConnectionContext(httpsServer))
        } else {
          management.start()
        }

        ClusterBootstrap(system).start()
        if (useXlCluster()) {
          XlCluster.init(xlrConfig.cluster)
          assert(XlCluster(system).membershipManagement != null)
        }
        managementFuture.onComplete {
          case Success(_) => initialize(system)
          case Failure(e) =>
            logger.error("Pekko management failed to start", e)
            SpringApplication.exit(applicationContext, () => NODE_DOWNED_EXIT_CODE)
        }
      case _ => throw new UnsupportedOperationException(s"Unsupported cluster mode \"${xlrConfig.cluster.mode}\" for Pekko cluster manager")
    }
  }

  private def initialize(system: ActorSystem): Unit = {
    try {
      // create new unmanaged actor every time system is initialized
      val unmanagedClusterListener = system.actorOf(actorSystemHolder.props(classOf[ClusterEventsListenerActor]), "clusterEventsListenerActor")
      unmanagedClusterListener ! Register
    } catch {
      case e: Exception =>
        logger.error("Failed to initialize cluster", e)
        SpringApplication.exit(applicationContext, () => NODE_DOWNED_EXIT_CODE)
    }
  }

  private def useXlCluster(): Boolean = {
    val downingProvider = xlrConfig.pekko.pekkoCluster.downingProviderClass
    if (XL_DOWNING_PROVIDERS.contains(downingProvider)) {
      true
    } else {
      false
    }
  }

  private def sslConfig: Config = xlrConfig.pekko.pekkoNativeConfig.getOptionalConfig("pekko.management.ssl").getOrElse(ConfigFactory.empty)

  private def useSsl: Boolean = sslConfig.getOptionalBoolean("enabled").getOrElse(false)

  private def getHttpsConnectionContext(): HttpsConnectionContext = {
    val sslContextConfig = SslContextConfig.fromConfig(sslConfig)
    val sslContext = SslContextFactory.getContext(sslContextConfig)
    ConnectionContext.httpsServer(sslContext)
  }
}

case class XlrStandaloneNodeBootstrap(actorSystemHolder: ActorSystemHolder, auxiliaryExecutor: ScheduledExecutorService) extends XlrActorSystemBootstrap {

  override def boot(): Unit = {
    actorSystemHolder.createActorSystem()
  }

  override def awaitStart(): Unit = {
    //no-op
  }
}

