package com.xebialabs.xlrelease.authentication


import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.security.XLReleasePermissions.isAdmin
import com.xebialabs.xlrelease.service.Users
import com.xebialabs.xlrelease.service.accountlock.AccountLockProtectionService
import grizzled.slf4j.Logging
import org.springframework.context.event.EventListener
import org.springframework.security.authentication.event.{AuthenticationFailureBadCredentialsEvent, AuthenticationFailureLockedEvent}
import org.springframework.stereotype.Component

import scala.jdk.CollectionConverters._

/**
 * Event listener that integrates account lock protection with Spring Security authentication events.
 *
 * This listener:
 * - Tracks failed login attempts when authentication fails
 * - Logs security events for monitoring
 */
@Component
class AuthenticationFailureEventListener(accountLockProtectionService: AccountLockProtectionService,
                                         usersRepository: Users,
                                         xlrConfig: XlrConfig) extends Logging {
  private val isAccountLockoutEnabled = xlrConfig.isAccountLockoutEnabled
  private val whitelistUsers: Set[String] = xlrConfig.whiteListUser.asScala.toSet

  /**
   * Handles authentication failure events
   * Records failed attempts for username
   *
   * @param event The authentication failure event
   */
  @EventListener
  def onAuthenticationFailure(event: AuthenticationFailureBadCredentialsEvent): Unit = {
    val authentication = event.getAuthentication
    val exception = event.getException
    val username = authentication.getName
    logger.debug(s"Authentication failure for user [${event.getAuthentication.getName}]. Reason: ${exception.getMessage}")
    if (isAccountLockoutEnabled && !whitelistUsers.contains(username) && !isAdmin(username)) {
      authentication.getPrincipal match {
        case _ if usersRepository.userExistsInRepository(authentication.getName) =>
          logger.debug(s"Account lockout is enabled.Processing account lockout checks for Internal user. ${username}")
          accountLockProtectionService.recordFailedAttempt(username)
        case _ => ()
          logger.debug(s"skipping account lock protection checks for user ${username} as it is not an internal user.")
      }
    } else {
      logger.debug(s"Account lockout is disabled or the user ${username} is whitelisted  Skipping protection checks.")
    }
  }

  @EventListener
  def onAuthenticationFailureLockedEvent(event: AuthenticationFailureLockedEvent): Unit = {
    val authentication = event.getAuthentication
    val exception = event.getException
    val username = authentication.getName
    logger.debug(s"Authentication failure for user [${event.getAuthentication.getName}]. Reason: ${exception.getMessage}")
    if (isAccountLockoutEnabled && !whitelistUsers.contains(username) && !isAdmin(username)) {
      authentication.getPrincipal match {
        case _ if usersRepository.userExistsInRepository(authentication.getName) =>
          logger.debug(s"Account lockout is enabled. Publishing authentication failure Lockout audit event for internal user: ${username}")
          accountLockProtectionService.publishAuthenticationFailureLockEvent(username)
        case _ => ()
          logger.debug(s"skipping reset account lock protection checks for user ${username} as it is not an internal user.")
      }
    } else {
      logger.debug(s"Account lockout is disabled or the user ${username} is whitelisted, skipping lock audit.")
    }
  }

}
