package com.xebialabs.xlrelease.calendar;

import java.util.Date;

import net.fortuna.ical4j.model.Calendar;
import net.fortuna.ical4j.model.ParameterList;
import net.fortuna.ical4j.model.component.VEvent;
import net.fortuna.ical4j.model.property.CalScale;
import net.fortuna.ical4j.model.property.Description;
import net.fortuna.ical4j.model.property.ProdId;
import net.fortuna.ical4j.model.property.Uid;
import net.fortuna.ical4j.model.property.Version;

import org.joda.time.DateTime;
import org.joda.time.Days;

import com.xebialabs.xlrelease.domain.Release;

import static com.google.common.base.Preconditions.checkState;
import static net.fortuna.ical4j.model.property.CalScale.VALUE_GREGORIAN;
import static net.fortuna.ical4j.model.property.Version.VALUE_2_0;

public class ReleaseCalendar {

    private final String id;
    private final String title;
    private final Date scheduledStartDate;
    private final Date dueDate;
    private final String description;

    private Calendar calendar;

    public ReleaseCalendar(Release release) {
        checkState(release.getScheduledStartDate() != null && release.getDueDate() != null, "Can't generate ICS file for release '%s', because it does not have start date or due date.", release.getTitle());

        this.id = release.getId();
        this.title = release.getTitle();
        this.scheduledStartDate = release.getScheduledStartDate();
        this.dueDate = release.getDueDate();
        this.description = release.getDescription();

        generateCalendar();
    }

    private void generateCalendar() {
        calendar = getCalendar();
        VEvent releaseEvent = getReleaseEvent();

        // Setting an UID allows outlook to update the event if previously imported
        releaseEvent.add(new Uid(id));

        calendar.add(releaseEvent);
    }

    private Calendar getCalendar() {
        Calendar calendar = new Calendar();
        calendar.add(new ProdId("-//Digital.ai Release//iCal4j 1.0//EN"));
        calendar.add(new Version(new ParameterList(), VALUE_2_0));
        calendar.add(new CalScale(VALUE_GREGORIAN));
        return calendar;
    }

    private VEvent getReleaseEvent() {
        DateTime startDate = new DateTime(scheduledStartDate);
        DateTime endDate = new DateTime(dueDate);

        int durationInDays = Days.daysBetween(startDate.withTimeAtStartOfDay(), endDate.withTimeAtStartOfDay()).getDays();

        VEvent releaseEvent;
        if (durationInDays < 2) {
            releaseEvent = new VEvent(scheduledStartDate.toInstant().atZone(java.time.ZoneId.systemDefault()),
                    dueDate.toInstant().atZone(java.time.ZoneId.systemDefault()), title);
        } else {
            // An event from oct 5 to oct 7 will span in the ICS  file from oct 5 to oct 8 because ICS uses midnight as the start of the day
            endDate = endDate.plusDays(1);
            releaseEvent = new VEvent(scheduledStartDate.toInstant().atZone(java.time.ZoneId.systemDefault()),
                    endDate.toDate().toInstant().atZone(java.time.ZoneId.systemDefault()), title);
        }

        releaseEvent.add(new Description(description));

        return releaseEvent;
    }

    @Override
    public String toString() {
        return calendar.toString();
    }
}
