package com.xebialabs.xlrelease.api.internal.converters

import com.xebialabs.xlrelease.configuration.{CustomLogoContentType, CustomLogoSettings}
import com.xebialabs.xlrelease.views.CustomLogoSettingsView
import org.jsoup.Jsoup
import org.jsoup.safety.Safelist
import org.springframework.stereotype.Component

import java.io.{ByteArrayOutputStream, InputStream}
import java.nio.charset.StandardCharsets
import java.util.Base64
import scala.util.Using

@Component
class CustomLogoSettingsConverter {
  private val svgSafeList: Safelist = Safelist.basic()
    .addTags("svg", "g", "path", "rect", "circle", "ellipse", "line", "polyline", "polygon", "text", "tspan")
    .addAttributes(":all", "style", "fill", "stroke", "d", "width", "height", "viewBox", "x", "y", "r", "cx", "cy", "points", "transform")
    .addProtocols("xlink:href", "http", "https")

  def from(fileName: String, contentType: String, inputStream: InputStream): CustomLogoSettings = {
    var customLogoSettings: CustomLogoSettings = null

    Using.resource(inputStream) { imageInFile =>
      var in = -1
      val buffer = new ByteArrayOutputStream
      while ( {
        in = imageInFile.read
        in != -1
      }) {
        buffer.write(in)
      }
      val imageData = buffer.toByteArray
      imageInFile.read(imageData)

      val sanitizedContent: Array[Byte] = sanitizeImage(contentType, imageData)

      customLogoSettings = new CustomLogoSettings
      customLogoSettings.setFilename(fileName)
      customLogoSettings.setContentType(contentType)
      customLogoSettings.setContent(Base64.getEncoder.encodeToString(sanitizedContent))
    }

    customLogoSettings
  }

  def toView(customLogoSettings: CustomLogoSettings): CustomLogoSettingsView = {
    val imageByteArray = Base64.getDecoder.decode(customLogoSettings.getContent)
    new CustomLogoSettingsView(customLogoSettings.getFilename, customLogoSettings.getContentType, imageByteArray)
  }

  private def sanitizeImage(contentType: String, imageData: Array[Byte]): Array[Byte] = {
    val sanitizedContent = if (contentType == CustomLogoContentType.IMAGE_SVG_XML.value()) {
      val imageContent = sanitizeSvg(new String(imageData, StandardCharsets.UTF_8))
      imageContent.getBytes(StandardCharsets.UTF_8)
    } else {
      imageData
    }
    sanitizedContent
  }

  private def sanitizeSvg(svgContent: String): String = {
    Jsoup.clean(svgContent, "", svgSafeList, new org.jsoup.nodes.Document.OutputSettings().prettyPrint(false))
  }

}
