package com.xebialabs.xlrelease.api.v1.impl

import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN
import com.xebialabs.xlrelease.api.v1.TenantApi
import com.xebialabs.xlrelease.api.v1.forms.{TenantLinkRequest, TenantRequest}
import com.xebialabs.xlrelease.configuration.TenantLimit
import com.xebialabs.xlrelease.domain.Tenant
import com.xebialabs.xlrelease.domain.status.TenantStatus
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.service.{TenantLimitService, TenantService}
import io.micrometer.core.annotation.Timed
import org.springframework.stereotype.Controller
import org.springframework.util.StringUtils.hasText

import java.util.{List => JList}
import scala.jdk.CollectionConverters._

@Controller
class TenantApiImpl(permissions: PermissionChecker,
                    tenantService: TenantService,
                    tenantLimitService: TenantLimitService)
  extends TenantApi {

  @Timed
  override def getTenants: JList[Tenant] = {
    permissions.check(ADMIN)
    tenantService.getAllTenants().asJava
  }

  @Timed
  override def getTenant(tenantId: String): Tenant = {
    permissions.check(ADMIN)
    tenantService.findTenantById(tenantId)
  }

  @Timed
  override def create(tenantRequest: TenantRequest): Tenant = {
    checkArgument(tenantRequest != null, "tenantRequest cannot be null")
    permissions.check(ADMIN)
    tenantService.createTenant(tenantRequest.getTenantName, TenantStatus.fromString(tenantRequest.getTenantStatus))
  }

  @Timed
  override def update(tenantId: String, tenantRequest: TenantRequest): Unit = {
    checkArgument(hasText(tenantId), "Tenant ID cannot be empty")
    checkArgument(tenantRequest != null, "tenantRequest cannot be null")
    permissions.check(ADMIN)
    val toUpdate = Tenant(tenantId, tenantRequest.getTenantName, TenantStatus.fromString(tenantRequest.getTenantStatus))
    tenantService.updateTenant(toUpdate)
  }

  @Timed
  override def delete(tenantId: String): Unit = {
    permissions.check(ADMIN)
    tenantService.deleteTenant(tenantId)
  }

  @Timed
  override def getLimits: JList[TenantLimit] = {
    permissions.check(ADMIN)
    tenantLimitService.getAllLimits().asJava
  }

  @Timed
  override def getLimit(tenantId: String): TenantLimit = {
    permissions.check(ADMIN)
    tenantLimitService.findLimitByTenantId(tenantId)
  }

  @Timed
  override def createLimit(tenantId: String, limit: TenantLimit): TenantLimit = {
    permissions.check(ADMIN)
    tenantLimitService.createLimit(tenantId, limit)
  }

  @Timed
  override def updateLimit(tenantId: String, limit: TenantLimit): Unit = {
    permissions.check(ADMIN)
    tenantLimitService.updateLimit(tenantId, limit)
  }

  @Timed
  override def deleteLimit(tenantId: String): Unit = {
    permissions.check(ADMIN)
    tenantLimitService.deleteLimit(tenantId)
  }

  @Timed
  override def linkResource(tenantId: String, linkRequests: JList[TenantLinkRequest]): Unit = {
    permissions.check(ADMIN)
    tenantService.linkResource(tenantId, linkRequests)
  }
}
