package com.xebialabs.xlrelease.security

import com.xebialabs.deployit.engine.api.dto
import com.xebialabs.deployit.engine.api.dto.Paging
import com.xebialabs.deployit.engine.spi.event.{RoleDeletedEvent, RolePermissionsChangedEvent, RolePrincipalsChangedEvent, RoleRenamedEvent}
import com.xebialabs.deployit.event.EventBusHolder
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.security.Role
import com.xebialabs.xlrelease.api.v1.filters.RolePrincipalsFilters
import com.xebialabs.xlrelease.domain.events.GlobalRolesOrPermissionsUpdatedEvent
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.security.PermissionChecker.GLOBAL_SECURITY_ALIAS
import com.xebialabs.xlrelease.security.ReleaseRoleService._
import com.xebialabs.xlrelease.service.{CiIdService, TeamService}
import com.xebialabs.xlrelease.views.RolePrincipalsView
import io.micrometer.core.annotation.Timed
import org.springframework.data.domain.{Page, Pageable}

import java.util
import scala.annotation.varargs
import scala.jdk.CollectionConverters._

class DefaultReleaseRoleService(xlReleaseEventBus: XLReleaseEventBus,
                                ciIdService: CiIdService,
                                releaseRoleRepository: ReleaseRoleRepository,
                                releaseRolePermissionRepository: ReleaseRolePermissionRepository
                               ) extends ReleaseRoleService {

  @Timed
  override def getGlobalRoles(rolePattern: String, paging: Paging, order: dto.Ordering): util.List[Role] = {
    releaseRoleRepository.getGlobalRoles(rolePattern, paging, order)
  }

  @Timed
  override def getGlobalRolePrincipalViews(filters: RolePrincipalsFilters, pageable: Pageable): Page[RolePrincipalsView] = {
    releaseRoleRepository.getGlobalRolePrincipals(filters, pageable)
  }

  @varargs
  @Timed
  override def create(onConfigurationItem: String, roles: Role*): Unit = {
    releaseRoleRepository.create(onConfigurationItem, roles: _*)
    fireRolePrincipalChangeEvents(roles: _*)
  }

  @varargs
  @Timed
  override def update(onConfigurationItem: String, roles: Role*): Unit = {
    releaseRoleRepository.update(onConfigurationItem, roles: _*)
    fireRolePrincipalChangeEvents(roles: _*)
  }

  @Timed
  override def renameGlobalRole(oldName: String, newName: String): Unit = {
    releaseRoleRepository.renameGlobalRole(oldName, newName)

    EventBusHolder.publish(new RoleRenamedEvent(oldName, newName))
    xlReleaseEventBus.publish(GlobalRolesOrPermissionsUpdatedEvent())
  }

  @Timed
  override def deleteGlobalRole(roleName: String): Unit = {
    val foundRole = releaseRoleRepository.getGlobalRole(roleName).getOrElse(
      throw new NotFoundException("Could not find the role [%s]", roleName)
    )
    releaseRoleRepository.deleteGlobalRole(foundRole.getId)

    EventBusHolder.publish(new RoleDeletedEvent(roleName))
    xlReleaseEventBus.publish(GlobalRolesOrPermissionsUpdatedEvent())
  }

  @Timed
  override def getGlobalRole(roleName: String): Option[Role] = {
    releaseRoleRepository.getGlobalRole(roleName)
  }

  @Timed
  override def getGlobalRolePrincipalsView(roleName: String): RolePrincipalsView = {
    releaseRoleRepository.getGlobalRolePrincipal(roleName).getOrElse(throw new NotFoundException(s"Role [$roleName] not found"))
  }

  @Timed
  override def globalRoleExists(roleName: String): Boolean = {
    releaseRoleRepository.globalRoleExists(roleName)
  }

  @varargs
  @Timed
  override def createGlobalRoleAndRolePermissions(requests: RoleCreateRequest*): Unit = {
    val (roles, rolePermissions) = requests.map { request =>
      val roleId = Ids.getName(ciIdService.getUniqueId("Role", TeamService.GLOBAL_ROLES_ROOT))
      val role = new Role(roleId, request.roleName, request.rolePrincipals.asScala.toList.asJava)
      val rolePermission = RolePermissionCreateRequest(roleId, role.getName, request.rolePermissions);
      (role, rolePermission)
    }.unzip

    // TODO inserting roles and their permissions should be single transactional operation
    //  rewrite to use spring
    create(roles: _*)
    createGlobalRolePermissions(rolePermissions: _*)

    xlReleaseEventBus.publish(GlobalRolesOrPermissionsUpdatedEvent())
  }

  @varargs
  @Timed
  override def updateGlobalRoleAndRolePermissions(requests: RoleUpdateRequest*): Unit = {
    val (roles, rolePermissionRequests) = requests.map { request =>
      val role = new Role(request.roleId, request.roleName, request.rolePrincipals.asScala.toList.asJava)
      val rolePermission = RolePermissionUpdateRequest(request.roleId, role.getName, request.rolePermissions)
      (role, rolePermission)
    }.unzip

    // TODO inserting roles and their permissions should be single transactional operation
    //  rewrite to use spring
    update(roles: _*)
    updateGlobalRolePermissions(rolePermissionRequests: _*)

    xlReleaseEventBus.publish(GlobalRolesOrPermissionsUpdatedEvent())
  }

  @Timed
  override def getGlobalRolePermissions(roleIds: RoleId*): GlobalRolePermissions = {
    releaseRolePermissionRepository.getGlobalRolePermissions(roleIds).collect {
      case RolePermission(roleId, rps) => (roleId, rps)
    }.toMap.asJava
  }

  @Timed
  override def getRolePermissions(roleId: RoleId): PermissionNames = {
    // TODO not sure about using permissionEditor
    releaseRolePermissionRepository.getRolePermissions(roleId).asJava
  }

  @varargs
  @Timed
  override def createGlobalRolePermissions(requests: RolePermissionCreateRequest*): Unit = {
    // TODO check would permission editor delete 1st? can we just add new ones
    val rolePermissions = requests.map { request => RolePermission(request.roleId, request.permissionNames) }
    releaseRolePermissionRepository.createGlobalRolePermissions(rolePermissions: _*)
    fireRolePermissionsChangedEvent(requests: _*)
  }

  @varargs
  @Timed
  override def updateGlobalRolePermissions(requests: RolePermissionUpdateRequest*): Unit = {
    // TODO check would permission editor delete 1st? can we just update
    val rolePermissions = requests.map { request => RolePermission(request.roleId, request.permissionNames) }
    releaseRolePermissionRepository.updateGlobalRolePermissions(rolePermissions: _*)
    fireRolePermissionsChangedEvent(requests: _*)
  }

  private def fireRolePermissionsChangedEvent(requests: RolePermissionChange*): Unit = {
    val msgs = requests.map { request => s"${request.roleName} => ${request.permissionNames}" }
    EventBusHolder.publish(new RolePermissionsChangedEvent(GLOBAL_SECURITY_ALIAS, msgs.asJava))
  }

  private def fireRolePrincipalChangeEvents(roles: Role*): Unit = {
    val rolePrincipals = roles.map(role => s"${role.getName} => ${role.getPrincipals}")
    EventBusHolder.publish(new RolePrincipalsChangedEvent(rolePrincipals.asJava))
  }
}
