package com.xebialabs.xlrelease.service

import com.xebialabs.xlrelease.service.TeamUpdateOperations.OperationMap.{OpType, OperationMap}
import com.xebialabs.xlrelease.service.TeamUpdateOperations.ReduceInstruction.{Append, DropLast, Noop, ReplaceLast}
import com.xebialabs.xlrelease.service.TeamUpdateOperations._
import com.xebialabs.xlrelease.views.teams._

import scala.collection.immutable.HashMap

object TeamUpdateOperations {

  def reduce(requests: Seq[TeamUpdateRequest]): Seq[TeamUpdateOperation] = {
    val operations = requests.foldLeft(TeamUpdateOperations()) {
      case (operations, request) => operations.process(request)
    }
    operations.operations()
  }

  object ReduceInstruction extends Enumeration {
    type ReduceInstruction = Value
    val Noop, ReplaceLast, DropLast, Append = Value
  }

  import ReduceInstruction._

  case class TeamOperationResult(reduceInstruction: ReduceInstruction, newOperation: Option[TeamUpdateOperation])

  object OperationMap {
    type TeamName = String
    type OpType = TeamUpdateOperation
    type OperationMap[OpType] = HashMap[TeamName, Seq[OpType]]

    def empty[OpType](): OperationMap[OpType] = {
      new OperationMap[OpType]
    }
  }
}

case class TeamUpdateOperations private(breakInheritance: Option[BreakInheritanceOperation] = None,
                                        operationMap: OperationMap[_ <: TeamUpdateOperation] = OperationMap.empty()) {

  def operations(): Seq[TeamUpdateOperation] = {
    Seq.empty ++ breakInheritance ++ operationMap.values.flatten.toSeq
  }

  //noinspection ScalaStyle
  def process(request: TeamUpdateRequest): TeamUpdateOperations = {
    request match {
      case BreakInheritance(containerId) =>
        this.copy(breakInheritance = Some(BreakInheritanceOperation(containerId)))
      case CreateTeam(containerId, teamName) =>
        updateOperationMap(containerId, teamName, None, createTeam(containerId, teamName))
      case DeleteTeam(containerId, teamId, teamName) =>
        updateOperationMap(containerId, teamName, None, deleteTeam(containerId, teamName, teamId))
      case AddRoleMember(containerId, teamId, teamName, roleName) =>
        updateOperationMap(containerId, teamName, teamId, addRole(roleName))
      case DeleteRoleMember(containerId, teamId, teamName, roleName) =>
        updateOperationMap(containerId, teamName, teamId, deleteRole(roleName))
      case AddPrincipalMember(containerId, teamId, teamName, principalName) =>
        updateOperationMap(containerId, teamName, teamId, addPrincipal(principalName))
      case DeletePrincipalMember(containerId, teamId, teamName, principalName) =>
        updateOperationMap(containerId, teamName, teamId, deletePrincipal(principalName))
      case AddTeamPermission(containerId, teamId, teamName, permission) =>
        updateOperationMap(containerId, teamName, teamId, addPermission(permission))
      case DeleteTeamPermission(containerId, teamId, teamName, permission) =>
        updateOperationMap(containerId, teamName, teamId, deletePermission(permission))
    }
  }

  private def updateOperationMap(containerId: String,
                                 teamName: String,
                                 teamId: Option[String],
                                 action: TeamUpdateOperation => TeamOperationResult): TeamUpdateOperations = {
    val newOperationMap = operationMap.updatedWith(teamName) {
      optionalOps => {
        val ops = optionalOps.getOrElse(Seq.empty[OpType])
        val lastOperation: TeamUpdateOperation = if (ops.nonEmpty) {
          ops.last
        } else {
          val opTeamId = if (breakInheritance.nonEmpty) None else teamId
          UpdateTeamOperation(containerId, opTeamId, teamName)
        }
        val result = action(lastOperation)
        reduceOps(ops, result)
      }
    }
    this.copy(operationMap = newOperationMap)
  }

  private def reduceOps(ops: Seq[TeamUpdateOperation], result: TeamOperationResult): Option[Seq[TeamUpdateOperation]] = {
    val TeamOperationResult(instruction, optionalOp) = result
    instruction match {
      case Noop => Some(ops)
      case ReplaceLast => Some(optionalOp.fold(ops.dropRight(1))(op => ops.dropRight(1) :+ op))
      case DropLast => Some(ops.dropRight(1))
      case Append => Some(optionalOp.fold(ops)(op => ops :+ op))
    }
  }

  private def createTeam(containerId: String, teamName: String)(teamUpdateOperation: TeamUpdateOperation): TeamOperationResult = {
    teamUpdateOperation match {
      case lastOp: CreateTeamOperation =>
        TeamOperationResult(ReduceInstruction.Noop, None)
      case _ =>
        TeamOperationResult(ReduceInstruction.Append, Some(CreateTeamOperation(containerId, teamName)))
    }
  }

  private def deleteTeam(containerId: String, teamName: String, teamId: Option[String])(teamUpdateOperation: TeamUpdateOperation): TeamOperationResult = {
    teamUpdateOperation match {
      case lastOp: CreateTeamOperation =>
        TeamOperationResult(ReduceInstruction.DropLast, None)
      case lastOp: UpdateTeamOperation =>
        TeamOperationResult(ReduceInstruction.ReplaceLast, Some(DeleteTeamOperation(containerId, teamId, team = teamName)))
      case lastOp: DeleteTeamOperation =>
        TeamOperationResult(ReduceInstruction.ReplaceLast, Some(DeleteTeamOperation(containerId, teamId, team = teamName)))
      case lastOp: BreakInheritanceOperation =>
        TeamOperationResult(ReduceInstruction.Append, Some(DeleteTeamOperation(containerId, teamId, team = teamName)))
    }
  }

  private def addRole(roleName: String)(teamUpdateOperation: TeamUpdateOperation): TeamOperationResult = {
    teamUpdateOperation match {
      case op: CreateTeamOperation =>
        val membership = op.membership.copyWith(role = Some(roleName))
        val newOp = op.copy(membership = membership)
        TeamOperationResult(ReplaceLast, Some(newOp))
      case op: UpdateTeamOperation =>
        val hadMembershipRemoved = op.removedMembership.roles.contains(roleName)
        val newRemovedMembership = op.removedMembership.copyWithout(role = Some(roleName))
        val newAddedMembership = if (hadMembershipRemoved) {
          op.addedMembership
        } else {
          op.addedMembership.copyWith(role = Some(roleName))
        }
        val newOp = op.copy(addedMembership = newAddedMembership, removedMembership = newRemovedMembership)
        val instruction = if (newOp.isEmpty) DropLast else ReplaceLast
        TeamOperationResult(instruction, Some(newOp))
      case op: DeleteTeamOperation =>
        val membership = TeamMembership(roles = Set(roleName))
        val newOp = CreateTeamOperation(containerId = op.containerId, team = op.team, membership = membership)
        TeamOperationResult(Append, Some(newOp))
      case op: BreakInheritanceOperation =>
        TeamOperationResult(Noop, None)
    }
  }

  private def deleteRole(roleName: String)(teamUpdateOperation: TeamUpdateOperation): TeamOperationResult = {
    teamUpdateOperation match {
      case op: CreateTeamOperation =>
        val membership = op.membership.copyWithout(role = Some(roleName))
        val newOp = op.copy(membership = membership)
        TeamOperationResult(ReplaceLast, Some(newOp))
      case op: UpdateTeamOperation =>
        val hadRole = op.addedMembership.roles.contains(roleName)
        val addedMembership = op.addedMembership.copyWithout(role = Some(roleName))
        val removedMembership = if (hadRole) {
          op.removedMembership
        } else {
          op.removedMembership.copyWith(role = Some(roleName))
        }
        val newOp = op.copy(addedMembership = addedMembership, removedMembership = removedMembership)
        val instruction = if (newOp.isEmpty) DropLast else ReplaceLast
        TeamOperationResult(instruction, Some(newOp))
      case op: DeleteTeamOperation =>
        TeamOperationResult(Noop, None)
      case op: BreakInheritanceOperation =>
        TeamOperationResult(Noop, None)
    }
  }

  private def addPrincipal(principalName: String)(teamUpdateOperation: TeamUpdateOperation): TeamOperationResult = {
    teamUpdateOperation match {
      case op: CreateTeamOperation =>
        val membership = op.membership.copyWith(principal = Some(principalName))
        val newOp = op.copy(membership = membership)
        TeamOperationResult(ReplaceLast, Some(newOp))
      case op: UpdateTeamOperation =>
        val hadMembershipRemoved = op.removedMembership.principals.contains(principalName)
        val newRemovedMembership = op.removedMembership.copyWithout(principal = Some(principalName))
        val newAddedMembership = if (hadMembershipRemoved) {
          op.addedMembership
        } else {
          op.addedMembership.copyWith(principal = Some(principalName))
        }
        val newOp = op.copy(addedMembership = newAddedMembership, removedMembership = newRemovedMembership)
        val instruction = if (newOp.isEmpty) DropLast else ReplaceLast
        TeamOperationResult(instruction, Some(newOp))
      case op: DeleteTeamOperation =>
        val membership = TeamMembership(principals = Set(principalName))
        val newOp = CreateTeamOperation(containerId = op.containerId, team = op.team, membership = membership)
        TeamOperationResult(Append, Some(newOp))
      case op: BreakInheritanceOperation =>
        TeamOperationResult(Noop, None)
    }
  }

  private def deletePrincipal(principalName: String)(teamUpdateOperation: TeamUpdateOperation): TeamOperationResult = {
    teamUpdateOperation match {
      case op: CreateTeamOperation =>
        val membership = op.membership.copyWithout(principal = Some(principalName))
        val newOp = op.copy(membership = membership)
        TeamOperationResult(ReplaceLast, Some(newOp))
      case op: UpdateTeamOperation =>
        val hadMembership = op.addedMembership.principals.contains(principalName)
        val addedMembership = op.addedMembership.copyWithout(principal = Some(principalName))
        val removedMembership = if (hadMembership) {
          op.removedMembership
        } else {
          op.removedMembership.copyWith(principal = Some(principalName))
        }
        val newOp = op.copy(addedMembership = addedMembership, removedMembership = removedMembership)
        val instruction = if (newOp.isEmpty) DropLast else ReplaceLast
        TeamOperationResult(instruction, Some(newOp))
      case op: DeleteTeamOperation =>
        TeamOperationResult(Noop, None)
      case op: BreakInheritanceOperation =>
        TeamOperationResult(Noop, None)
    }
  }

  private def addPermission(permissionName: String)(teamUpdateOperation: TeamUpdateOperation): TeamOperationResult = {
    teamUpdateOperation match {
      case op: CreateTeamOperation =>
        val membership = op.membership.copyWith(permission = Some(permissionName))
        val newOp = op.copy(membership = membership)
        TeamOperationResult(ReplaceLast, Some(newOp))
      case op: UpdateTeamOperation =>
        val hadMembershipRemoved = op.removedMembership.permissions.contains(permissionName)
        val newRemovedMembership = op.removedMembership.copyWithout(permission = Some(permissionName))
        val newAddedMembership = if (hadMembershipRemoved) {
          op.addedMembership
        } else {
          op.addedMembership.copyWith(permission = Some(permissionName))
        }
        val newOp = op.copy(addedMembership = newAddedMembership, removedMembership = newRemovedMembership)
        val instruction = if (newOp.isEmpty) DropLast else ReplaceLast
        TeamOperationResult(instruction, Some(newOp))
      case op: DeleteTeamOperation =>
        val membership = TeamMembership(permissions = Set(permissionName))
        val newOp = CreateTeamOperation(containerId = op.containerId, team = op.team, membership = membership)
        TeamOperationResult(Append, Some(newOp))
      case op: BreakInheritanceOperation =>
        TeamOperationResult(Noop, None)
    }
  }

  private def deletePermission(permissionName: String)(teamUpdateOperation: TeamUpdateOperation): TeamOperationResult = {
    teamUpdateOperation match {
      case op: CreateTeamOperation =>
        val membership = op.membership.copyWithout(permission = Some(permissionName))
        val newOp = op.copy(membership = membership)
        TeamOperationResult(ReplaceLast, Some(newOp))
      case op: UpdateTeamOperation =>
        val hadMembership = op.addedMembership.permissions.contains(permissionName)
        val addedMembership = op.addedMembership.copyWithout(permission = Some(permissionName))
        val removedMembership = if (hadMembership) {
          op.removedMembership
        } else {
          op.removedMembership.copyWith(permission = Some(permissionName))
        }
        val newOp = op.copy(addedMembership = addedMembership, removedMembership = removedMembership)
        val instruction = if (newOp.isEmpty) DropLast else ReplaceLast
        TeamOperationResult(instruction, Some(newOp))
      case op: DeleteTeamOperation =>
        TeamOperationResult(Noop, None)
      case op: BreakInheritanceOperation =>
        TeamOperationResult(Noop, None)
    }
  }
}