package com.xebialabs.xlrelease.storage.service

import com.xebialabs.xlrelease.storage.Storage
import com.xebialabs.xlrelease.storage.domain.{StorageEntry, StorageEntryRef, StorageScheme}
import grizzled.slf4j.Logging

import java.io.InputStream
import java.net.URI
import scala.util.Using

class DefaultStorageService(storages: Array[Storage], override val defaultStorageType: String) extends StorageService with Logging {

  private def getStorage(storageScheme: StorageScheme): Storage = {
    val uriScheme = storageScheme.uriScheme match {
      case Some(uriScheme) => uriScheme
      case None => defaultStorageType
    }
    storages.find(s => s.uriScheme == uriScheme)
      .getOrElse(throw new IllegalArgumentException(s"Unable to find storage for entry $uriScheme"))
  }

  override def store(storageEntry: StorageEntry): URI = {
    val storage = getStorage(storageEntry)
    val entryUri = new URI(storage.uriScheme, null, storageEntry.uriPath, null)
    logger.trace(s"Storing log entry: $storageEntry with uri $entryUri")
    Using(storageEntry.content) { is =>
      val storedUri = storage.put(entryUri, is)
      storedUri
    }.recoverWith {
      case t => throw new RuntimeException(s"Unable to store log entry $storageEntry with uri $entryUri", t)
    }.get
  }

  override def get(storageEntryRef: StorageEntryRef): InputStream = {
    val storage = getStorage(storageEntryRef)
    storage.get(storageEntryRef.uri)
  }

  override def listDirectories(storageEntryRef: StorageEntryRef): List[URI] = {
    val storage = getStorage(storageEntryRef)
    storage.listDirectories(storageEntryRef.uri)
  }

  override def listFiles(storageEntryRef: StorageEntryRef): List[URI] = {
    val storage = getStorage(storageEntryRef)
    storage.listFiles(storageEntryRef.uri)
  }

  override def delete(storageEntryRef: StorageEntryRef): Boolean = {
    val storage = getStorage(storageEntryRef)
    storage.delete(storageEntryRef.uri)
  }

  override def deleteIfEmpty(storageEntryRef: StorageEntryRef): Boolean = {
    val storage = getStorage(storageEntryRef)
    storage.deleteIfEmpty(storageEntryRef.uri)
  }
}
