import React, {Component} from 'react';
import ReactDOM from 'react-dom';
import escapeRegExp from 'lodash/escapeRegExp';
import includes from 'lodash/includes';
import isEqual from 'lodash/isEqual';
import noop from 'lodash/noop';
import uniq from 'lodash/uniq';
import map from 'lodash/map';
import {Suggestions} from './suggestions.component';
import PropTypes from 'prop-types';
import ClassNames from 'classnames';
import {Tag} from './tag.component';
import {keyCodes} from '../../../constants/key-codes';
import {DEFAULT_CLASSNAMES, DEFAULT_LABEL_FIELD, DEFAULT_PLACEHOLDER} from './xlr-tags-constants';

import './xlr-tags.less';

/**
 * @description
 * This component has been inspired from react-tags component (https://github.com/prakhar1989/react-tags/tree/v6.4.2)
 * It is customised as per XebiaLabs requirements.
 *
 * For all options, see https://github.com/prakhar1989/react-tags/tree/v6.4.2#options
 */

export class XlrTags extends Component {
    static propTypes = {
        allowAddFromPaste: PropTypes.bool,
        allowAddOnlyFromSuggestion: PropTypes.bool,
        allowDeleteFromEmptyInput: PropTypes.bool,
        allowUnique: PropTypes.bool,
        autocomplete: PropTypes.oneOfType([PropTypes.bool, PropTypes.number]),
        autofocus: PropTypes.bool,
        classNames: PropTypes.object,
        delimiters: PropTypes.arrayOf(PropTypes.number),
        handleAddition: PropTypes.func,
        handleDelete: PropTypes.func,
        handleFilterSuggestions: PropTypes.func,
        handleInputBlur: PropTypes.func,
        handleInputChange: PropTypes.func,
        handleInputFocus: PropTypes.func,
        handleTagClick: PropTypes.func,
        id: PropTypes.string,
        inputValue: PropTypes.string,
        labelField: PropTypes.string,
        maxLength: PropTypes.number,
        minQueryLength: PropTypes.number,
        name: PropTypes.string,
        placeholder: PropTypes.string,
        readOnly: PropTypes.bool,
        removeComponent: PropTypes.func,
        renderSuggestion: PropTypes.func,
        resetInputOnDelete: PropTypes.bool,
        shouldRenderSuggestions: PropTypes.func,
        suggestions: PropTypes.arrayOf(
            PropTypes.shape({
                id: PropTypes.string.isRequired,
            })
        ),
        tags: PropTypes.arrayOf(
            PropTypes.shape({
                id: PropTypes.string.isRequired,
                readOnly: PropTypes.bool.isRequired,
                className: PropTypes.string,
            })
        )
    };

    static defaultProps = {
        placeholder: DEFAULT_PLACEHOLDER,
        labelField: DEFAULT_LABEL_FIELD,
        suggestions: [],
        delimiters: [keyCodes.ENTER],
        autofocus: true,
        handleDelete: noop,
        handleAddition: noop,
        allowAddOnlyFromSuggestion: false,
        allowDeleteFromEmptyInput: true,
        allowAddFromPaste: true,
        resetInputOnDelete: true,
        autocomplete: false,
        readOnly: false,
        allowUnique: true,
        tags: []
    };

    constructor(props) {
        super(props);

        const {suggestions} = props;
        this.state = {
            suggestions,
            query: '',
            isFocused: false,
            selectedIndex: -1,
            selectionMode: false,
            shouldShowSuggestionFromBottom: false
        };
        this.handleFocus = this.handleFocus.bind(this);
        this.handleBlur = this.handleBlur.bind(this);
        this.handleKeyDown = this.handleKeyDown.bind(this);
        this.handleChange = this.handleChange.bind(this);
        this.handlePaste = this.handlePaste.bind(this);
        this.resetAndFocusInput = this.resetAndFocusInput.bind(this);
        this.handleSuggestionHover = this.handleSuggestionHover.bind(this);
        this.handleSuggestionClick = this.handleSuggestionClick.bind(this);
    }

    componentDidMount() {
        const {autofocus, readOnly} = this.props;

        if (autofocus && !readOnly) {
            this.resetAndFocusInput();
        }
    }

    componentDidUpdate(prevProps) {
        if (!isEqual(prevProps.suggestions, this.props.suggestions)) {
            this.updateSuggestions();
        }
    }

    filteredSuggestions(query, suggestions) {
        if (this.props.handleFilterSuggestions) {
            return this.props.handleFilterSuggestions(query, suggestions);
        }

        const exactSuggestions = suggestions.filter((item) => {
            return this.getQueryIndex(query, item) === 0 && !includes(map(this.props.tags, this.props.labelField), item[this.props.labelField]);
        });
        const partialSuggestions = suggestions.filter((item) => {
            return this.getQueryIndex(query, item) > 0 && !includes(map(this.props.tags, this.props.labelField), item[this.props.labelField]);
        });
        return exactSuggestions.concat(partialSuggestions);
    }

    getQueryIndex(query, item) {
        return item[this.props.labelField]
            .toLowerCase()
            .indexOf(query.toLowerCase());
    }

    resetAndFocusInput() {
        this.setState({query: ''});
        if (this.textInput) {
            this.textInput.value = '';
            this.textInput.focus();
        }
    }

    handleDelete(i, e) {
        this.props.handleDelete(i, e);
        if (!this.props.resetInputOnDelete) {
            this.textInput && this.textInput.focus();
        } else {
            this.resetAndFocusInput();
        }
        e.stopPropagation();
    }

    handleTagClick(i, e) {
        if (this.props.handleTagClick) {
            this.props.handleTagClick(i, e);
        }
        if (!this.props.resetInputOnDelete) {
            this.textInput && this.textInput.focus();
        } else {
            this.resetAndFocusInput();
        }
    }

    handleChange(e) {
        if (this.props.handleInputChange) {
            this.props.handleInputChange(e.target.value);
        }

        const query = e.target.value.trim();

        this.setState({query}, this.updateSuggestions);
    }

    /**
     * Convert an array of delimiter characters into a regular expression
     * that can be used to split content by those delimiters.
     * @param {Array<char>} delimiters Array of characters to turn into a regex
     * @returns {RegExp} Regular expression
     */
    buildRegExpFromDelimiters(delimiters) {
        const delimiterChars = delimiters
            .map((delimiter) => {
                // See: http://stackoverflow.com/a/34711175/1463681
                const chrCode = delimiter - 48 * Math.floor(delimiter / 48);
                return String.fromCharCode(96 <= delimiter ? chrCode : delimiter);
            })
            .join('');
        const escapedDelimiterChars = escapeRegExp(delimiterChars);
        return new RegExp(`[${escapedDelimiterChars}]+`);
    }

    updateSuggestions() {
        const {query, selectedIndex} = this.state;
        const suggestions = this.filteredSuggestions(query, this.props.suggestions);

        this.setState({
            suggestions,
            selectedIndex:
                selectedIndex >= suggestions.length
                    ? suggestions.length - 1
                    : selectedIndex,
        });
    }

    isOutOfViewport (elem) {
        const bounding = elem.getBoundingClientRect();
        return bounding.bottom + 150 > window.innerHeight;//150px is max height for suggestion component
    }

    handleFocus(e) {
        const value = e.target.value;
        if (this.props.handleInputFocus) {
            this.props.handleInputFocus(value);
        }
        const isOutOfViewport = this.isOutOfViewport(this.textInput);
        this.setState({isFocused: true, shouldShowSuggestionFromBottom: isOutOfViewport});
    }

    handleBlur(e) {
        const value = e.target.value;
        if (this.props.handleInputBlur) {
            this.props.handleInputBlur(value);
            if (this.textInput) {
                this.textInput.value = '';
            }
        }
        this.setState({isFocused: false});
    }

    handleKeyDown(e) {
        const {query, selectedIndex, suggestions, selectionMode} = this.state;

        // hide suggestions menu on escape
        if (e.keyCode === keyCodes.ESC) {
            e.preventDefault();
            e.stopPropagation();
            this.setState({
                selectedIndex: -1,
                selectionMode: false,
                suggestions: [],
            });
        }

        // When one of the terminating keys is pressed, add current query to the tags.
        // If no text is typed in so far, ignore the action - so we don't end up with a terminating
        // character typed in.
        if (includes(this.props.delimiters, e.keyCode) && !e.shiftKey) {
            if (e.keyCode !== keyCodes.TAB || query !== '') {
                e.preventDefault();
            }

            const selectedQuery =
                selectionMode && selectedIndex !== -1
                    ? suggestions[selectedIndex]
                    : {id: query, [this.props.labelField]: query};

            if (selectedQuery !== '') {
                this.addTag(selectedQuery);
            }
        }

        // when backspace key is pressed and query is blank, delete tag
        if (
            e.keyCode === keyCodes.BACKSPACE &&
            query === '' &&
            this.props.allowDeleteFromEmptyInput
        ) {
            this.handleDelete(this.props.tags.length - 1, e);
        }

        // up arrow
        if (e.keyCode === keyCodes.UP) {
            e.preventDefault();
            this.setState({
                selectedIndex:
                    selectedIndex <= 0 ? suggestions.length - 1 : selectedIndex - 1,
                selectionMode: true,
            });
        }

        // down arrow
        if (e.keyCode === keyCodes.DOWN) {
            e.preventDefault();
            this.setState({
                selectedIndex:
                    suggestions.length === 0
                        ? -1
                        : (selectedIndex + 1) % suggestions.length,
                selectionMode: true,
            });
        }
    }

    handlePaste(e) {
        if (!this.props.allowAddFromPaste) {
            return;
        }

        e.preventDefault();

        const clipboardData = e.clipboardData || window.clipboardData;
        const clipboardText = clipboardData.getData('text');

        const {maxLength = clipboardText.length} = this.props;

        const maxTextLength = Math.min(maxLength, clipboardText.length);
        const pastedText = clipboardData.getData('text').substr(0, maxTextLength);

        // Used to determine how the pasted content is split.
        const delimiterRegExp = this.buildRegExpFromDelimiters(this.props.delimiters);
        const tags = pastedText.split(delimiterRegExp);

        // Only add unique tags
        uniq(tags).forEach((tag) =>
            this.addTag({id: tag, [this.props.labelField]: tag})
        );
    }

    highlight(tag) {
        const classNames = {...DEFAULT_CLASSNAMES, ...this.props.classNames};
        /* eslint-disable react/no-find-dom-node */
        const node = ReactDOM.findDOMNode(this);
        /* eslint-enable react/no-find-dom-node */
        if (node instanceof HTMLElement) {
            const parent = node.querySelector(`.${classNames.tag}[data-tag-id='${tag.id}']`).parentNode;
            parent.classList.remove(classNames.tagHighlight);
            setTimeout(() => parent.classList.add(classNames.tagHighlight), 100);
        }
    }

    addTag(tag) {
        const {tags, labelField, allowUnique} = this.props;
        tag.readOnly = false; // by default, will be overwritten on  tag = possibleMatches[0];
        if (!tag.id || !tag[labelField]) {
            return;
        }
        const existingKeys = tags.map(item => item.id?.toLowerCase());

        // Return if tag has been already added
        if (allowUnique && includes(existingKeys, tag.id?.toLowerCase())) {
            this.highlight(tag);
            return;
        }
        if (this.props.autocomplete) {
            const possibleMatches = this.filteredSuggestions(
                tag[labelField],
                this.props.suggestions
            );

            if (
                (this.props.autocomplete === 1 && possibleMatches.length === 1) ||
                (this.props.autocomplete === true && possibleMatches.length)
            ) {
                tag = possibleMatches[0];
            }
        }

        // Return if entered tag is any arbitrary value and not selected from suggestions
        if (this.props.allowAddOnlyFromSuggestion &&
            !includes(this.props.suggestions.map(item => item.id.toLowerCase()), tag.id.toLowerCase())) {
            return;
        }

        // call method to add
        this.props.handleAddition(tag);

        // reset the state
        this.setState({
            query: '',
            selectionMode: false,
            selectedIndex: -1,
        });

        this.resetAndFocusInput();
    }

    handleSuggestionClick(i) {
        this.addTag(this.state.suggestions[i]);
    }

    handleSuggestionHover(i) {
        this.setState({
            selectedIndex: i,
            selectionMode: true,
        });
    }

    getTagItems() {
        const {
            tags,
            labelField,
            removeComponent,
        } = this.props;

        const classNames = {...DEFAULT_CLASSNAMES, ...this.props.classNames};

        return tags.map((tag, index) => {
            return (
                <div className={classNames.selected} key={`div-${tag.id}-${index}`}>
                    <Tag
                        classNames={classNames}
                        displayIcon={!tag.isVirtual}
                        index={index}
                        key={`${tag.id}-${index}`}
                        labelField={labelField}
                        onDelete={this.handleDelete.bind(this, index)}
                        onTagClicked={this.handleTagClick.bind(this, index)}
                        readOnly={tag.readOnly}
                        removeComponent={removeComponent}
                        tag={tag}
                    />
                </div>
            );
        });
    }

    render() {
        const tagItems = this.getTagItems();
        const classNames = {...DEFAULT_CLASSNAMES, ...this.props.classNames};

        // get the suggestions for the given query
        const query = this.state.query.trim(),
            selectedIndex = this.state.selectedIndex,
            suggestions = this.state.suggestions;

        const {
            placeholder,
            name: inputName,
            id: inputId,
            maxLength
        } = this.props;

        const tagInput = !this.props.readOnly ? (
            <div className={classNames.tagInput}>
                <input
                    aria-label={placeholder}
                    className={classNames.tagInputField}
                    id={inputId}
                    maxLength={maxLength}
                    name={inputName}
                    onBlur={this.handleBlur}
                    onChange={this.handleChange}
                    onFocus={this.handleFocus}
                    onKeyDown={this.handleKeyDown}
                    onPaste={this.handlePaste}
                    placeholder={placeholder}
                    ref={(input) => {
                        this.textInput = input;
                    }}
                    type="text"
                    value={this.props.inputValue}
                />

                <Suggestions
                    classNames={classNames}
                    handleClick={this.handleSuggestionClick}
                    handleHover={this.handleSuggestionHover}
                    isFocused={this.state.isFocused}
                    isShowFromBottom={this.state.shouldShowSuggestionFromBottom}
                    labelField={this.props.labelField}
                    minQueryLength={this.props.minQueryLength}
                    query={query}
                    renderSuggestion={this.props.renderSuggestion}
                    selectedIndex={selectedIndex}
                    shouldRenderSuggestions={this.props.shouldRenderSuggestions}
                    suggestions={suggestions}
                />
            </div>
        ) : null;

        return (
            <div className={ClassNames(classNames.tags, 'react-tags-wrapper')}>
                {tagItems}
                {tagInput}
            </div>
        );
    }
}

