import React from 'react';
import Functor from '../functor';
import PropTypes from 'prop-types';
import pluralize from 'pluralize';
import colors from '../../../js/colors';

const inputNames = ['months', 'days', 'hours'];

export class DurationPicker extends React.Component {

    static propTypes = {
        description: PropTypes.string,
        disabled: PropTypes.bool.isRequired,
        hours: PropTypes.number.isRequired,
        label: PropTypes.string,
        minHours: PropTypes.number,
        onModelChange: PropTypes.func.isRequired,
        onPristinityChange: PropTypes.func,
        placeholder: PropTypes.string,
        tip: PropTypes.string
    };

    static getDerivedStateFromProps(props, state) {
        if (props.hours !== state.originalValue) {
            const {months, days, hours} = mapHoursToValues(props.hours);
            return {
                originalValue: Number(props.hours),
                months,
                days,
                hours,
                editMode: false
            };
        }
        return null;
    }

    constructor(props) {
        super(props);
        this.state = {};
    }

    handleOnBlur = e => {
        if (!(e.relatedTarget !== null && inputNames.includes(e.relatedTarget.name) ||
            document.activeElement !== null && inputNames.includes(document.activeElement.name))) {
            this.switchMode(e, true);
        }
    };
    handleKeyDown = e => {
        if (e.key === 'Escape') {
            this.switchMode(e);
        }
        if (e.key === 'Enter') {
            this.switchMode(e, true);
        }
    };

    handleEnter = e => {
        if (e.key === 'Enter') {
            this.switchMode(e);
        }
    };

    editMode = () => {
        return (
            <div className="duration-editor" style={styles.editStyle}>
                <input
                    autoFocus
                    name="months"
                    onBlur={this.handleOnBlur}
                    onChange={this.onChange}
                    onKeyDown={this.handleKeyDown}
                    style={styles.inputStyle}
                    type="text"
                    value={this.state.months}
                />
                <label style={styles.labelStyle}>months</label>
                <input name="days"
                       onBlur={this.handleOnBlur}
                       onChange={this.onChange}
                       onKeyDown={this.handleKeyDown}
                       style={styles.inputStyle}
                       type="text"
                       value={this.state.days}
                />
                <label style={styles.labelStyle}>days</label>
                <input
                    name="hours"
                    onBlur={this.handleOnBlur}
                    onChange={this.onChange}
                    onKeyDown={this.handleKeyDown}
                    style={styles.inputStyle}
                    type="text"
                    value={this.state.hours}
                />
                <label style={styles.labelStyle}>hours</label>
            </div>
        );
    };

    validateMinimum = (name, value, min) => {
        const measures = ['days', 'hours', 'months'];
        measures.splice(measures.indexOf(name), 1);
        const found = measures.find(m => {
            const val = this.state[m];
            return val > 0 && val >= min[m];
        });
        return value > 0 && value >= min[name] || !!found;
    };

    onChange = evt => {
        const {minHours} = this.props;
        const target = evt.target;
        const name = target.name;
        const value = target.value;
        let isMoreThanMin = value >= 0;
        if (minHours) {
            isMoreThanMin = this.validateMinimum(name, value, mapHoursToValues(minHours));
        }
        if (isNumeric(value) && (isMoreThanMin && value <= 99)) {
            this.setState({
                [name]: Math.max(0, value)
            });
        } else if (value === '') {
            this.setState({
                [name]: ''
            });
        }
    };

    switchMode = (evt, set = false) => {
        const {onModelChange, onPristinityChange} = this.props;
        const flag = !this.state.editMode;
        if (set) {
            const {months, days, hours} = extractValues(this.state);
            const rs = evaluateDuration(months, days, hours);
            onModelChange(rs);
            if (onPristinityChange) {
                onPristinityChange(rs === this.state.originalValue);
            }
            this.setState({
                editMode: flag
            });
        } else {
            const {months, days, hours} = mapHoursToValues(this.state.originalValue);
            this.setState({
                months,
                days,
                hours,
                editMode: flag
            });
        }
    };

    viewMode = () => {
        let {months, days, hours} = extractValues(this.state);
        if (!months && !days && !hours) {
            const originals = mapHoursToValues(this.state.originalValue);
            months = originals.months;
            days = originals.days;
            hours = originals.hours;
        }
        const view = new Functor("")
            .map(s => appendString(s, months, pluralize('month', months)))
            .map(s => appendString(s, days, pluralize('day', days)))
            .map(s => appendString(s, hours, pluralize('hour', hours)))
            .map(s => this.props.tip ? `${s} ${this.props.tip}` : s)
            .map(s => s === "" ? this.props.placeholder || "Placeholder" : s)
            .identity();
        return !this.props.disabled ? (
            <input
                onClick={this.switchMode}
                onKeyDown={this.handleEnter}
                readOnly
                style={styles.viewStyle}
                value={view}
            />
        ) : (<input
            readOnly
            style={styles.viewDisabledStyle}
            value={view}
        />);
    };

    render() {
        const {description, disabled, label} = this.props;

        return (
            <div className="xl-react-components">
                {label && <div className="xl-react-components-label">{label}</div>}
                <div className={`xl-react-components-input ${disabled && 'is-disabled'}`}>
                    {this.state.editMode ? this.editMode() : this.viewMode()}
                    <div className="xl-react-components-description">
                        <div>{description}</div>
                    </div>
                </div>

            </div>
        );
    }
}

function extractValues(state) {
    const {months, days, hours} = state;
    return {
        months: defaultVal(months),
        days: defaultVal(days),
        hours: defaultVal(hours)
    };
}

function defaultVal(val) {
    return isNumeric(val) ? val : 0;
}

function isNumeric(n) {
    return !isNaN(parseFloat(n)) && isFinite(n);
}

function mapHoursToValues(hours) {
    const m = Math.floor(hours / 720);
    const remainder = hours - (m * 720);
    const d = Math.floor(remainder / 24);
    const h = hours - (m * 720) - (d * 24);
    return {
        months: m,
        days: d,
        hours: h,
    };
}

function evaluateDuration(months, days, hours) {
    return months * 720 + days * 24 + hours;
}

function appendString(str, value, marker) {
    return (value !== 0) ? `${str} ${value} ${marker}` : str;
}

const styles = {
    editStyle: {
        display: "flex",
        flexDirection: "row",
        flexWrap: "nowrap",
        alignItems: "center",
    },
    viewDisabledStyle: {
        backgroundColor: colors.background,
        borderWidth: 1,
        borderStyle: "solid",
        borderColor: colors.lightgray,
        borderRadius: 4,
        minHeight: 30,
        display: "flex",
        alignItems: "center",
        paddingLeft: 10,
        color: colors.gray,
        fontSize: 12,
        width: "100%"
    },
    viewStyle: {
        borderWidth: 1,
        borderStyle: "solid",
        borderColor: colors.gray,
        borderRadius: 4,
        minHeight: 30,
        display: "flex",
        alignItems: "center",
        paddingLeft: 10,
        color: colors.gray,
        fontSize: 12,
        width: "100%"
    },
    labelStyle: {
        fontSize: 11,
        cursor: "default",
        marginBottom: 0
    },
    inputStyle: {
        marginLeft: 5,
        marginRight: 5,
        marginBottom: 0,
        minHeight: 30,
        minWidth: 40,
        width: 40,
        borderRadius: 4,
    },
    buttonStyle: {
        marginLeft: 5,
        marginRight: 5,
        marginBottom: 0,
        marginTop: 0,
        minHeight: 30,
        minWidth: 50,
        width: 50,
        borderRadius: 4,
        fontSize: 11
    }
};

