import React from 'react';
import PropTypes from 'prop-types';
import colors from '../../../js/colors';

import './avatar.less';

const defaultColors = [
    {background: colors.green, text: colors.white},
    {background: colors.blue, text: colors.white},
    {background: colors.orange, text: colors.black},
    {background: colors.purple, text: colors.white},
    {background: colors.yellow, text: colors.black},
    {background: colors.red, text: colors.white},
    {background: colors.antracite, text: colors.white},
    {background: colors.lightgray, text: colors.black}
];

export const AvatarComponent = ({text, title = ''}) => {
    const parsedText = () => {
        const textArray = text.split(' ');

        if (textArray.length > 1) {
            const firstInitial = textArray[0].slice(0, 1).toUpperCase();
            const secondInitial = textArray[1].slice(0, 1).toUpperCase();
            return `${firstInitial}${secondInitial}`;
        } else {
            return text ? text.slice(0, 1).toUpperCase() : '';
        }
    };

    // https://en.wikipedia.org/wiki/Linear_congruential_generator
    const stringAsciiPRNG = (value, m) => {
        // Xn+1 = (a * Xn + c) % m
        // 0 < a < m
        // 0 <= c < m
        // 0 <= X0 < m

        const charCodes = [...value].map(letter => letter.charCodeAt(0));
        const len = charCodes.length;

        const a = (len % (m - 1)) + 1;
        const c = charCodes.reduce((current, next) => current + next) % m;

        let random = charCodes[0] % m;
        for (let i = 0; i < len; i++)
            random = ((a * random) + c) % m;

        return random;
    };

    const getRandomColor = (value) => {
        if (!value)
            return {background: 'transparent', color: 'transparent'};

        const colorIndex = stringAsciiPRNG(value, defaultColors.length);
        return defaultColors[colorIndex];
    };

    return (
        <span className='avatar'
              style={{background: getRandomColor(text).background, color: getRandomColor(text).text}}
              title={title}>
        {parsedText()}
    </span>
    );
};

AvatarComponent.propTypes = {
    text: PropTypes.string.isRequired,
    title: PropTypes.string
};
