import React, { Component, Fragment } from 'react';
import PropTypes from 'prop-types';
import isEqual from 'lodash/isEqual';
import isFunction from 'lodash/isFunction';

const maybeScrollSuggestionIntoView = (suggestionEl, suggestionsContainer) => {
    const containerHeight = suggestionsContainer.offsetHeight;
    const suggestionHeight = suggestionEl.offsetHeight;
    const relativeSuggestionTop = suggestionEl.offsetTop - suggestionsContainer.scrollTop;

    if (relativeSuggestionTop + suggestionHeight >= containerHeight) {
        suggestionsContainer.scrollTop += relativeSuggestionTop - containerHeight + suggestionHeight;
    } else if (relativeSuggestionTop < 0) {
        suggestionsContainer.scrollTop += relativeSuggestionTop;
    }
};

export class Suggestions extends Component {
    static propTypes = {
        classNames: PropTypes.object,
        handleClick: PropTypes.func.isRequired,
        handleHover: PropTypes.func.isRequired,
        isFocused: PropTypes.bool.isRequired,
        isShowFromBottom: PropTypes.bool.isRequired,
        labelField: PropTypes.string.isRequired,
        minQueryLength: PropTypes.number,
        query: PropTypes.string.isRequired,
        renderSuggestion: PropTypes.func,
        selectedIndex: PropTypes.number.isRequired,
        shouldRenderSuggestions: PropTypes.func,
        suggestions: PropTypes.array.isRequired,
    };

    static defaultProps = {
        minQueryLength: 2,
    };

    shouldComponentUpdate(nextProps) {
        const { props } = this;
        const shouldRenderSuggestions = props.shouldRenderSuggestions || this.shouldRenderSuggestions;
        return (
            props.isFocused !== nextProps.isFocused ||
            !isEqual(props.suggestions, nextProps.suggestions) ||
            shouldRenderSuggestions(nextProps.query) ||
            shouldRenderSuggestions(nextProps.query) !== shouldRenderSuggestions(props.query)
        );
    }

    componentDidUpdate(prevProps) {
        const { selectedIndex, classNames } = this.props;

        if (this.suggestionsContainer && prevProps.selectedIndex !== selectedIndex) {
            const activeSuggestion = this.suggestionsContainer.querySelector(classNames.activeSuggestion);

            if (activeSuggestion) {
                maybeScrollSuggestionIntoView(activeSuggestion, this.suggestionsContainer);
            }
        }
    }

    shouldRenderSuggestions = (query) => {
        const { minQueryLength, isFocused } = this.props;
        return query.length >= minQueryLength && isFocused;
    };

    renderSuggestion = (item, query) => {
        const { renderSuggestion } = this.props;
        if (isFunction(renderSuggestion)) {
            return renderSuggestion(item, query);
        }
        const { [this.props.labelField]: labelValue } = item;
        return <span>{labelValue}</span>;
    };

    render() {
        const { props } = this;

        const suggestions = props.suggestions.map((item, i) => {
            return (
                <li
                    className={i === props.selectedIndex ? props.classNames.activeSuggestion : ''}
                    key={i}
                    onMouseDown={props.handleClick.bind(null, i)}
                    onMouseOver={props.handleHover.bind(null, i)}
                    onTouchStart={props.handleClick.bind(null, i)}
                >
                    {this.renderSuggestion(item, props.query)}
                </li>
            );
        });

        const shouldRenderSuggestions = props.shouldRenderSuggestions || this.shouldRenderSuggestions;
        if (suggestions.length === 0 || !shouldRenderSuggestions(props.query)) {
            return <Fragment />;
        }

        return (
            <div
                className={this.props.classNames.suggestions}
                ref={(elem) => (this.suggestionsContainer = elem)}
                style={props.isShowFromBottom ? { bottom: '0px', position: 'fixed' } : null}
            >
                <ul> {suggestions} </ul>
            </div>
        );
    }
}
