import React from 'react';
import PropTypes from 'prop-types';
import { mount } from 'enzyme/build';
import { Tag } from './tag.component';

function mockItem(overrides) {
    const props = Object.assign(
        {},
        {
            tag: { id: '1', text: 'FooBar', className: 'action' },
            onDelete: jest.fn(),
            readOnly: false,
            classNames: {
                tag: 'tag',
                remove: 'remove',
            },
        },
        overrides,
    );
    return <Tag {...props} />;
}

describe('Tag component', () => {
    test('should show the classnames of children properly', () => {
        const tagComponent = mount(mockItem());
        expect(tagComponent.find('.tag').length).toBe(1);
        expect(tagComponent.text()).toBe('FooBar');
    });

    test('should show cross for removing tag when read-only is false', () => {
        const tagComponent = mount(mockItem());
        expect(tagComponent.find('a.remove').length).toBe(1);
    });

    test('should not show cross for removing tag when read-only is true', () => {
        const tagComponent = mount(mockItem({ readOnly: true }));
        expect(tagComponent.find('a.remove').length).toBe(0);
    });

    test('renders passed in removed component correctly', () => {
        const CustomRemoveComponent = function () {
            return <a className="remove">delete me</a>;
        };
        const tagComponent = mount(mockItem({ removeComponent: CustomRemoveComponent }));
        expect(tagComponent.find('a.remove').length).toBe(1);
        expect(tagComponent.text()).toContain('delete me');
    });

    test('renders conditionaly passed in removed component correctly', () => {
        const CustomConditionRemoveComponent = function (props) {
            return props.tag.id === '1' ? null : <a className="removeTag">x</a>;
        };
        CustomConditionRemoveComponent.propTypes = {
            tag: PropTypes.shape({
                id: PropTypes.string.isRequired,
            }),
        };
        const tagComponent = mount(mockItem({ removeComponent: CustomConditionRemoveComponent }));
        expect(tagComponent.find('.removeTag').length).toBe(0);
    });

    test('calls the delete handler correctly', () => {
        const spy = jest.fn();
        const tagComponent = mount(mockItem({ onDelete: spy }));
        tagComponent.find('a.remove').simulate('click');
        expect(spy).toHaveBeenCalledTimes(1);

        spy.mockReset();
        tagComponent.find('a.remove').simulate('keyDown');
        expect(spy).toHaveBeenCalledTimes(1);
    });

    test('should add className passed in tags to the tag', () => {
        const tagComponent = mount(mockItem());
        expect(tagComponent.find('.action').length).toBe(1);
    });

    test('calls the tag click handler correctly', () => {
        const spy = jest.fn();
        const tagComponent = mount(mockItem({ onTagClicked: spy }));
        tagComponent.find('span.tag-wrapper').simulate('click');
        expect(spy).toHaveBeenCalledTimes(1);

        spy.mockReset();
        tagComponent.find('span.tag-wrapper').simulate('keyDown');
        expect(spy).toHaveBeenCalledTimes(1);
    });

    test('should trigger the tag click handler on touchStart', () => {
        const spy = jest.fn();
        const tagComponent = mount(mockItem({ onTagClicked: spy }));
        tagComponent.find('span.tag-wrapper').simulate('touchStart');
        expect(spy).toHaveBeenCalledTimes(1);
    });
});
