import React from 'react';
import { ReactWrapper } from 'enzyme';
import { DotThemeProvider } from '@digital-ai/dot-components';
import { TextAreaMarkdown, TextareaMarkdownProps, VariablesInterpolatorFactory } from './textarea-markdown.component';
import { mountComponentWithStore } from '../../../../tests/unit/testing-utils';
import * as angularAccessor from '../../../features/common/services/angular-accessor';

type ActionTypeId = 'save-button' | 'cancel-button' | 'edit-button' | 'delete-button';
const getAngularServiceSpy = jest.spyOn(angularAccessor, 'default') as never as jest.SpyInstance<VariablesInterpolatorFactory, [name: any]>;

describe('TextAreaMarkdown Component', () => {
    const onTextChange = jest.fn();

    const defaultProps: TextareaMarkdownProps = {
        inputId: 'markdown-input',
        name: 'markdown',
        onTextChange,
        rows: 5,
        text: 'my **markdown** text',
    };

    const mount = (props: TextareaMarkdownProps = defaultProps) => {
        return mountComponentWithStore(
            <DotThemeProvider>
                <TextAreaMarkdown {...props} />
            </DotThemeProvider>,
        );
    };

    const markdownSwitcherSelector = '.markdown-switcher';
    const markdownViewerSelector = `${markdownSwitcherSelector} .markdown-viewer`;
    const markdownWrapperSelector = `${markdownViewerSelector} .markdown-wrapper p`;
    const selectAction = (testId: ActionTypeId) => `${markdownSwitcherSelector} button[data-testid="${testId}"]`;
    const textareaSelector = `${markdownSwitcherSelector} textarea#markdown-input`;

    const checkInitialState = (wrapper: ReactWrapper) => {
        expect(wrapper.find(textareaSelector).exists()).toBeFalsy();
        expect(wrapper.find(markdownWrapperSelector).text()).toBe('my markdown text');
        expect(wrapper.find(`${markdownWrapperSelector} strong`).text()).toBe('markdown');
    };

    const switchToEditMode = (wrapper: ReactWrapper) => {
        wrapper.find(selectAction('edit-button')).simulate('click');
        expect(wrapper.find(textareaSelector).exists()).toBeTruthy();
        expect(wrapper.find(markdownViewerSelector).exists()).toBeFalsy();
    };

    const switchToViewMode = (wrapper: ReactWrapper) => {
        wrapper.find(selectAction('cancel-button')).simulate('click');
        expect(wrapper.find(textareaSelector).exists()).toBeFalsy();
        expect(wrapper.find(markdownViewerSelector).exists()).toBeTruthy();
    };

    const selectMarkup = (markupSelector: string) => `${markdownWrapperSelector} ${markupSelector}`;

    const checkEmptyInitialState = (wrapper: ReactWrapper) => {
        expect(wrapper.find(textareaSelector).exists()).toBeFalsy();
        expect(wrapper.find(markdownWrapperSelector).exists()).toBeFalsy();
    };

    const checkActionButtons = (wrapper: ReactWrapper, shouldDeleteBtnExist = false) => {
        const actions = wrapper.find(`${markdownViewerSelector} .markdown-viewer-actions`);
        expect(actions.exists()).toBeTruthy();
        expect(actions.find('[data-testid="delete-button"]').exists()).toBe(shouldDeleteBtnExist);
        expect(actions.find('[data-testid="edit-button"]').exists()).toBeTruthy();
    };

    let interpolateInTextMock = jest.fn();

    beforeEach(() => {
        jest.resetAllMocks();
        // somehow i need to do this here, dunno why
        interpolateInTextMock = jest.fn().mockReturnValue(defaultProps.text);
        getAngularServiceSpy.mockReturnValue({
            interpolateInText: interpolateInTextMock,
        });
    });

    test('should render markdown by default', () => {
        const wrapper = mount();
        checkInitialState(wrapper);
        checkActionButtons(wrapper);
        expect(interpolateInTextMock).not.toHaveBeenCalled();
    });

    test('should render markdown with mentions', () => {
        const users = [
            { username: 'admin', fullName: 'Release Administrator' },
            { username: 'bob', fullName: 'Bob Builder' },
            { username: 'suzy', fullName: 'Suzy' },
            { username: 'ann', fullName: '' },
            { username: 'Itchy' },
            { username: 'Scratchy' },
        ];
        const text = `## My **Manual task**. @admin can you check?  
        
@ann assign @itchy and @unknown to task  
        
This should be solved by@Scratchy or by **@suzy**  
        
**Note: @bob is PO**`;
        const wrapper = mount({ ...defaultProps, users, enableMentions: true, text });
        expect(wrapper.find(textareaSelector).exists()).toBeFalsy();
        const markdownWrapper = wrapper.find(`${markdownViewerSelector} .markdown-wrapper`);

        expect(markdownWrapper.find('h2').exists()).toBeTruthy();
        expect(markdownWrapper.find('h2 strong').exists()).toBeTruthy();
        expect(markdownWrapper.find('h2 strong').text()).toBe('Manual task');
        expect(markdownWrapper.find('h2 span').text()).toBe('Release Administrator');
        expect(markdownWrapper.find('h2 span').prop('className')).toBe('mention');
        expect(markdownWrapper.find('h2 span').prop('title')).toBe('admin');

        const markdownParagraphs = markdownWrapper.find('p');
        expect(markdownParagraphs.exists()).toBeTruthy();
        expect(markdownParagraphs.length).toBe(3);

        expect(markdownParagraphs.at(0).text()).toBe('ann assign @itchy and @unknown to task');
        expect(markdownParagraphs.at(0).find('span').text()).toBe('ann');
        expect(markdownParagraphs.at(0).find('span').prop('className')).toBe('mention');
        expect(markdownParagraphs.at(0).find('span').prop('title')).toBe('ann');

        expect(markdownParagraphs.at(1).text()).toBe('This should be solved by@Scratchy or by Suzy');
        expect(markdownParagraphs.at(1).find('span').text()).toBe('Suzy');
        expect(markdownParagraphs.at(1).find('span').prop('className')).toBe('mention');
        expect(markdownParagraphs.at(1).find('span').prop('title')).toBe('suzy');

        expect(markdownParagraphs.at(2).text()).toBe('Note: Bob Builder is PO');
        expect(markdownParagraphs.at(2).find('strong').text()).toBe('Note: Bob Builder is PO');
        expect(markdownParagraphs.at(2).find('span').text()).toBe('Bob Builder');
        expect(markdownParagraphs.at(2).find('span').prop('className')).toBe('mention');
        expect(markdownParagraphs.at(2).find('span').prop('title')).toBe('bob');
    });

    test('should switch to edit mode', () => {
        const wrapper = mount();
        // initial markdown mode
        checkInitialState(wrapper);
        // edit mode
        switchToEditMode(wrapper);
        // markdown mode
        switchToViewMode(wrapper);
        expect(onTextChange).not.toHaveBeenCalled();
    });

    test('should render edit mode with mentions', () => {
        const wrapper = mount({ ...defaultProps, enableMentions: true });
        checkInitialState(wrapper);
        switchToEditMode(wrapper);
        expect(wrapper.find('MentionsInput').exists()).toBeTruthy();
    });

    test('should save and cancel changes', () => {
        const wrapper = mount();
        // original text
        checkInitialState(wrapper);

        // do change
        switchToEditMode(wrapper);
        wrapper.find(textareaSelector).simulate('change', { target: { value: 'my ```fancy code```' } });
        // cancel change
        switchToViewMode(wrapper);
        // original state
        checkInitialState(wrapper);
        expect(wrapper.find(selectMarkup('code')).exists()).toBeFalsy();
        // do change
        switchToEditMode(wrapper);
        wrapper.find(textareaSelector).simulate('change', { target: { value: 'my ```fancy code```' } });
        // save change
        wrapper.find(selectAction('save-button')).simulate('click');
        expect(onTextChange).toHaveBeenCalledWith('my ```fancy code```');
        expect(wrapper.find(markdownWrapperSelector).text()).toBe('my fancy code');
        expect(wrapper.find(selectMarkup('code')).text()).toBe('fancy code');
    });

    test('should render delete button when onDelete function is defined', () => {
        const wrapper = mount({ ...defaultProps, onDelete: jest.fn() });
        checkInitialState(wrapper);
        checkActionButtons(wrapper, true);
    });

    test('should not delete original text when clicking on delete', () => {
        const wrapper = mount({ ...defaultProps, onDelete: jest.fn() });
        expect(wrapper.find(markdownWrapperSelector).text()).toBe('my markdown text');
        wrapper.find('button[data-testid="delete-button"]').simulate('click');
        expect(wrapper.find(markdownWrapperSelector).text()).toBe('my markdown text');
    });

    test('should render empty placeholder when value is empty', () => {
        const wrapper = mount({ ...defaultProps, text: '' });
        checkEmptyInitialState(wrapper);
        checkActionButtons(wrapper);

        const placeholderWrapper = wrapper.find(`${markdownSwitcherSelector} .placeholder-wrapper`);
        expect(placeholderWrapper.exists()).toBeTruthy();
        expect(placeholderWrapper.find('DotTypography.placeholder').text()).toBe('');
    });

    test('should render placeholder with text when value is empty', () => {
        const wrapper = mount({ ...defaultProps, text: '', placeholder: 'Add text...' });
        checkEmptyInitialState(wrapper);
        checkActionButtons(wrapper);

        const placeholderWrapper = wrapper.find(`${markdownSwitcherSelector} .placeholder-wrapper`);
        expect(placeholderWrapper.exists()).toBeTruthy();
        expect(placeholderWrapper.find('DotTypography.placeholder').text()).toBe('Add text...');
    });

    test('should call replace variables', () => {
        const variables = { '${var}': 'variableValue' };
        const wrapper = mount({ ...defaultProps, variables });
        checkInitialState(wrapper);
        expect(interpolateInTextMock).toHaveBeenCalledWith(variables, defaultProps.text);
    });
});
