import React, { useEffect, useState } from 'react';
import PropTypes from 'prop-types';
import isEmpty from 'lodash/isEmpty';
import classNames from 'classnames';
import { TextAreaPassword } from './TextAreaPassword';

import './text-area-password-edit-mode.less';

const TextAreaPasswordEditMode = ({ name, onChange, readOnly = false, required = false, value }) => {
    const [isEditMode, setEditMode] = useState(false);
    const [passwordValue, setPasswordValue] = useState(false);

    useEffect(() => {
        setPasswordValue(value);
    }, [value]);

    const save = () => {
        onChange(passwordValue);
        setEditMode(false);
    };

    const cancel = () => {
        setEditMode(false);
    };

    const isValid = () => {
        if (required) {
            return !isEmpty(passwordValue);
        }
        return true;
    };

    const notEditMode = () => (
        <div className={classNames('display inline-textarea', { 'is-invalid': !isValid() })}>
            <span className="text placeholder light-text">Click on edit icon to modify the password</span>
            <i className="xl-icon edit-icon" onClick={() => setEditMode(true)} />
        </div>
    );

    const editMode = () => (
        <React.Fragment>
            <TextAreaPassword name={name} onChange={setPasswordValue} required={required} value={passwordValue} />
            <div className="text-area-password-action-buttons">
                <div className="ok" onClick={save}>
                    <i className="xl-icon check-icon" />
                    Save
                </div>
                <div className="cancel" onClick={cancel}>
                    <i className="xl-icon close-icon" />
                    Cancel
                </div>
            </div>
        </React.Fragment>
    );

    const readOnlyMode = () => (
        <div className="field-readonly read-only">
            <span className="light-text">{isEmpty(passwordValue) ? '' : '*****'}</span>
        </div>
    );

    return <div className="text-area-password-edit-mode">{readOnly ? readOnlyMode() : isEditMode ? editMode() : notEditMode()}</div>;
};

TextAreaPasswordEditMode.propTypes = {
    name: PropTypes.string,
    onChange: PropTypes.func.isRequired,
    readOnly: PropTypes.bool,
    required: PropTypes.bool,
    value: PropTypes.any,
};

export { TextAreaPasswordEditMode };
