import React from 'react';
import { ReactWrapper } from 'enzyme';
import constant from 'lodash/constant';
import { TextAreaMarkdown, TextareaMarkdownProps } from './textarea-markdown.component';
import { mountWithStoreAndTheme } from '../../../../tests/unit/testing-utils';
import * as angularAccessor from '../../../features/common/services/angular-accessor';
import { VariablesInterpolatorFactory } from '../../../features/tasks/types/angular';

type ActionTypeId = 'save-button' | 'cancel-button' | 'edit-button' | 'delete-button';
const getAngularServiceSpy = jest.spyOn(angularAccessor, 'default') as unknown as jest.SpyInstance<VariablesInterpolatorFactory, [name: unknown]>;

describe('TextAreaMarkdown Component', () => {
    const onTextChange = jest.fn();

    const defaultProps: TextareaMarkdownProps = {
        inputId: 'markdown-input',
        name: 'markdown',
        onTextChange,
        rows: 5,
        text: 'my **markdown** text',
    };

    const mount = (props: TextareaMarkdownProps = defaultProps) => mountWithStoreAndTheme(<TextAreaMarkdown {...props} />);

    const markdownSwitcherSelector = '.markdown-switcher';
    const markdownViewerSelector = `${markdownSwitcherSelector} .markdown-viewer`;
    const markdownWrapperSelector = `${markdownViewerSelector} .markdown-wrapper p`;
    const selectAction = (testId: ActionTypeId) => `${markdownSwitcherSelector} button[data-testid="${testId}"]`;
    const textareaSelector = `${markdownSwitcherSelector} textarea#markdown-input`;
    const isInEditMode = (wrapper: ReactWrapper) => wrapper.find(textareaSelector).exists();
    const getMarkdownText = (wrapper: ReactWrapper) => wrapper.find(markdownWrapperSelector).text();
    const getMarkDownWrapperSelector = (wrapper: ReactWrapper) => wrapper.find(markdownWrapperSelector);
    const getPlaceholderWrapper = (wrapper: ReactWrapper) => wrapper.find(`${markdownSwitcherSelector} .placeholder-wrapper`);
    const setMarkdownText = (wrapper: ReactWrapper, value: string) => wrapper.find(textareaSelector).simulate('change', { target: { value } });
    const saveChanges = (wrapper: ReactWrapper) => wrapper.find(selectAction('save-button')).simulate('click');
    const checkInitialState = (wrapper: ReactWrapper) => {
        expect(isInEditMode(wrapper)).toBeFalsy();
        expect(getMarkdownText(wrapper)).toBe('my markdown text');
        expect(wrapper.find(`${markdownWrapperSelector} strong`).text()).toBe('markdown');
    };

    const switchToEditMode = (wrapper: ReactWrapper) => {
        wrapper.find(selectAction('edit-button')).simulate('click');
        expect(isInEditMode(wrapper)).toBeTruthy();
        expect(wrapper.find(markdownViewerSelector).exists()).toBeFalsy();
    };

    const switchToViewMode = (wrapper: ReactWrapper) => {
        wrapper.find(selectAction('cancel-button')).simulate('click');
        expect(isInEditMode(wrapper)).toBeFalsy();
        expect(wrapper.find(markdownViewerSelector).exists()).toBeTruthy();
    };

    const selectMarkup = (markupSelector: string) => `${markdownWrapperSelector} ${markupSelector}`;

    const checkEmptyInitialState = (wrapper: ReactWrapper) => {
        expect(isInEditMode(wrapper)).toBeFalsy();
        expect(wrapper.find(markdownWrapperSelector).exists()).toBeFalsy();
    };

    const checkActionButtons = (wrapper: ReactWrapper, shouldDeleteBtnExist = false) => {
        const actions = wrapper.find(`${markdownViewerSelector} .markdown-viewer-actions`);
        expect(actions.exists()).toBeTruthy();
        expect(actions.find('[data-testid="delete-button"]').exists()).toBe(shouldDeleteBtnExist);
        expect(actions.find('[data-testid="edit-button"]').exists()).toBeTruthy();
    };

    let interpolateInTextMock = jest.fn();

    beforeEach(() => {
        jest.resetAllMocks();
        // somehow i need to do this here, dunno why
        interpolateInTextMock = jest.fn().mockReturnValue(defaultProps.text);
        getAngularServiceSpy.mockReturnValue({
            interpolateInText: interpolateInTextMock,
        });
    });

    it('should render markdown by default', () => {
        const wrapper = mount();
        checkInitialState(wrapper);
        checkActionButtons(wrapper);
        expect(interpolateInTextMock).not.toHaveBeenCalled();
    });

    it('should render markdown with mentions', () => {
        const users = [
            { username: 'admin', fullName: 'Release Administrator' },
            { username: 'bob', fullName: 'Bob Builder' },
            { username: 'suzy', fullName: 'Suzy' },
            { username: 'ann', fullName: '' },
            { username: 'Itchy' },
            { username: 'Scratchy' },
        ];
        const text = `## My **Manual task**. @admin can you check?  
        
@ann assign @itchy and @unknown to task  
        
This should be solved by@Scratchy or by **@suzy**  
        
**Note: @bob is PO**`;
        const wrapper = mount({ ...defaultProps, users, enableMentions: true, text });
        expect(wrapper.find(textareaSelector).exists()).toBeFalsy();
        const markdownWrapper = wrapper.find(`${markdownViewerSelector} .markdown-wrapper`);

        expect(markdownWrapper.find('h2').exists()).toBeTruthy();
        expect(markdownWrapper.find('h2 strong').exists()).toBeTruthy();
        expect(markdownWrapper.find('h2 strong').text()).toBe('Manual task');
        expect(markdownWrapper.find('h2 span').text()).toBe('Release Administrator');
        expect(markdownWrapper.find('h2 span').prop('className')).toBe('mention');
        expect(markdownWrapper.find('h2 span').prop('title')).toBe('admin');

        const markdownParagraphs = markdownWrapper.find('p');
        expect(markdownParagraphs.exists()).toBeTruthy();
        expect(markdownParagraphs.length).toBe(3);

        expect(markdownParagraphs.at(0).text()).toBe('ann assign @itchy and @unknown to task');
        expect(markdownParagraphs.at(0).find('span').text()).toBe('ann');
        expect(markdownParagraphs.at(0).find('span').prop('className')).toBe('mention');
        expect(markdownParagraphs.at(0).find('span').prop('title')).toBe('ann');

        expect(markdownParagraphs.at(1).text()).toBe('This should be solved by@Scratchy or by Suzy');
        expect(markdownParagraphs.at(1).find('span').text()).toBe('Suzy');
        expect(markdownParagraphs.at(1).find('span').prop('className')).toBe('mention');
        expect(markdownParagraphs.at(1).find('span').prop('title')).toBe('suzy');

        expect(markdownParagraphs.at(2).text()).toBe('Note: Bob Builder is PO');
        expect(markdownParagraphs.at(2).find('strong').text()).toBe('Note: Bob Builder is PO');
        expect(markdownParagraphs.at(2).find('span').text()).toBe('Bob Builder');
        expect(markdownParagraphs.at(2).find('span').prop('className')).toBe('mention');
        expect(markdownParagraphs.at(2).find('span').prop('title')).toBe('bob');
    });

    it('should switch to edit mode', () => {
        const wrapper = mount();
        // initial markdown mode
        checkInitialState(wrapper);
        // edit mode
        switchToEditMode(wrapper);
        // markdown mode
        switchToViewMode(wrapper);
        expect(onTextChange).not.toHaveBeenCalled();
    });

    it('should render edit mode with mentions', () => {
        const wrapper = mount({ ...defaultProps, enableMentions: true });
        checkInitialState(wrapper);
        switchToEditMode(wrapper);
        expect(wrapper.find('MentionsInput').exists()).toBeTruthy();
    });

    it('should save and cancel changes', () => {
        const wrapper = mount();
        // original text
        checkInitialState(wrapper);

        // do change
        switchToEditMode(wrapper);
        setMarkdownText(wrapper, 'my ```fancy code```');
        // cancel change
        switchToViewMode(wrapper);
        // original state
        checkInitialState(wrapper);
        expect(wrapper.find(selectMarkup('code')).exists()).toBeFalsy();
        // do change
        switchToEditMode(wrapper);
        setMarkdownText(wrapper, 'my ```fancy code```');
        // save change
        saveChanges(wrapper);
        expect(onTextChange).toHaveBeenCalledWith('my ```fancy code```');
        expect(getMarkdownText(wrapper)).toBe('my fancy code');
        expect(wrapper.find(selectMarkup('code')).text()).toBe('fancy code');
    });

    it('should not save empty spaces', () => {
        const wrapper = mount();
        // original text
        checkInitialState(wrapper);

        // do change
        switchToEditMode(wrapper);
        setMarkdownText(wrapper, '    ');
        saveChanges(wrapper);

        expect(isInEditMode(wrapper)).toBe(false);
        expect(onTextChange).not.toHaveBeenCalled();
        expect(getMarkdownText(wrapper)).toBe('my markdown text');
    });

    it('should render delete button when onDelete function is defined', () => {
        const wrapper = mount({ ...defaultProps, onDelete: jest.fn() });
        checkInitialState(wrapper);
        checkActionButtons(wrapper, true);
    });

    it('should not delete original text when clicking on delete', () => {
        const wrapper = mount({ ...defaultProps, onDelete: jest.fn() });
        expect(getMarkdownText(wrapper)).toBe('my markdown text');
        wrapper.find('button[data-testid="delete-button"]').simulate('click');
        expect(getMarkdownText(wrapper)).toBe('my markdown text');
    });

    it('should render empty placeholder when value is empty', () => {
        const wrapper = mount({ ...defaultProps, text: '' });
        checkEmptyInitialState(wrapper);
        checkActionButtons(wrapper);

        const placeholderWrapper = wrapper.find(`${markdownSwitcherSelector} .placeholder-wrapper`);
        expect(placeholderWrapper.exists()).toBeTruthy();
        expect(placeholderWrapper.find('DotTypography.placeholder').text()).toBe('');
    });

    it('should render placeholder with text when value is empty', () => {
        const wrapper = mount({ ...defaultProps, text: '', placeholder: 'Add text...' });
        checkEmptyInitialState(wrapper);
        checkActionButtons(wrapper);

        const placeholderWrapper = getPlaceholderWrapper(wrapper);
        expect(placeholderWrapper.exists()).toBeTruthy();
        expect(placeholderWrapper.find('DotTypography.placeholder').text()).toBe('Add text...');
    });

    it('should call replace variables', () => {
        const variables = { '${var}': 'variableValue' };
        const wrapper = mount({ ...defaultProps, variables });
        checkInitialState(wrapper);
        expect(interpolateInTextMock).toHaveBeenCalledWith(variables, defaultProps.text);
    });

    it('should revert changes when empty value is set but value is required', () => {
        const text = 'My text';
        const wrapper = mount({ ...defaultProps, text });
        switchToEditMode(wrapper);
        setMarkdownText(wrapper, '');
        saveChanges(wrapper);
        expect(getMarkdownText(wrapper)).toBe(text);
        expect(onTextChange).not.toHaveBeenCalled();
    });

    it('should show placeholder when empty value is set but value is not required', () => {
        const wrapper = mount({ ...defaultProps, text: 'this will be removed', isValueRequired: false });
        switchToEditMode(wrapper);
        setMarkdownText(wrapper, '');
        saveChanges(wrapper);
        expect(getMarkDownWrapperSelector(wrapper)).not.toExist();
        expect(getPlaceholderWrapper(wrapper)).toExist();
        expect(onTextChange).toHaveBeenCalledWith('');
    });

    it('should render links inside Release', () => {
        const link = `${document.location.origin}/#/releases`;
        const text = `[content](${link})`;
        const wrapper = mount({ ...defaultProps, text });
        const markdownWrapper = wrapper.find(markdownWrapperSelector);
        const anchor = markdownWrapper.find('a a');
        expect(anchor.prop('href')).toBe(link);
        expect(anchor.prop('target')).toBeUndefined();
        expect(anchor.prop('rel')).toBeUndefined();
        expect(anchor.text()).toBe('content');
    });

    it('should render links outside Release', () => {
        const link = `https://www.google.com`;
        const text = `[content](${link})`;
        const wrapper = mount({ ...defaultProps, text });
        const markdownWrapper = wrapper.find(markdownWrapperSelector);

        const anchor = markdownWrapper.find('a a');
        expect(anchor.prop('href')).toBe(link);
        expect(anchor.prop('target')).toBe('_blank');
        expect(anchor.prop('rel')).toBe('noreferrer noopener');
        expect(anchor.text()).toBe('content');
    });

    describe('when clicking on editor', () => {
        const clickOnViewer = (wrapper: ReactWrapper, target: HTMLElement) => {
            wrapper.find(markdownViewerSelector).simulate('mouseup', { target });
        };
        const doubleClickOnViewer = (wrapper: ReactWrapper, target: HTMLElement) => {
            wrapper.find(markdownViewerSelector).simulate('dblclick', { target });
        };

        it('should not go to edit mode if text is selected', () => {
            const getSelection = jest.fn();
            getSelection.mockReturnValue({
                toString: constant('my mocked selected text'),
            });
            window.document.getSelection = getSelection;

            const wrapper = mount(defaultProps);
            clickOnViewer(wrapper, document.createElement('div'));
            expect(isInEditMode(wrapper)).toBe(false);
        });

        it('should not go to edit mode if link is clicked', () => {
            const wrapper = mount(defaultProps);
            const fakeLink = document.createElement('a');
            fakeLink.setAttribute('href', 'www.9gag.com');
            clickOnViewer(wrapper, fakeLink);
            expect(isInEditMode(wrapper)).toBe(false);
        });

        it('should not go to edit mode if delete button is clicked', () => {
            const wrapper = mount(defaultProps);
            const fakeDotIcon = document.createElement('i');
            fakeDotIcon.setAttribute('class', 'dot-i');
            clickOnViewer(wrapper, fakeDotIcon);
            expect(isInEditMode(wrapper)).toBe(false);
        });

        it('should not go to edit mode if component is in readOnly', () => {
            const wrapper = mount({ ...defaultProps, isReadOnly: true });
            clickOnViewer(wrapper, document.createElement('div'));
            expect(isInEditMode(wrapper)).toBe(false);
        });

        it('should go to edit mode if viewer is clicked', () => {
            const wrapper = mount(defaultProps);
            doubleClickOnViewer(wrapper, document.createElement('div'));
            expect(isInEditMode(wrapper)).toBe(true);
        });
    });
});
