import React from 'react';
import { DotChip } from '@digital-ai/dot-components';
import { mockResizeObserver, mountWithTheme, ReactWrapper } from '../../../../tests/unit/testing-utils';
import { ChipGroup, ChipGroupProps } from './chip-group.component';
import * as hooks from './hooks/useChipGroup.hook';
import { UseChipGroupWidths } from './hooks/useChipGroup.hook';

describe('ChipGroup', () => {
    let wrapper: ReactWrapper;
    const labels = ['my long long long long long long long label', 'label2', 'label3', 'label4'];

    beforeAll(() => {
        mockResizeObserver();
    });

    afterEach(() => {
        jest.restoreAllMocks();
    });

    const defaultProps: ChipGroupProps = {
        labels,
    };

    const mount = (props = defaultProps) => {
        wrapper = mountWithTheme(<ChipGroup {...props} />);
    };

    const getChipGroupSpans = () => wrapper.find("span[data-testid='chip-group-span']");
    const getExceedingWidthChipSpan = () => wrapper.find("span[data-testid='exceeding-width-chip-span']");

    const assertExceedingWidthChipSpan = (isRendered: boolean, text?: string) => {
        const exceedingWidthChipSpan = getExceedingWidthChipSpan();

        if (!isRendered) {
            expect(exceedingWidthChipSpan).not.toExist();
        } else {
            expect(exceedingWidthChipSpan).toExist();
            if (text) {
                const exceedingWidthChip = exceedingWidthChipSpan.find(DotChip);
                expect(exceedingWidthChip).toExist();
                expect(exceedingWidthChip).toHaveText(text);
                expect(exceedingWidthChip.props().size).toBe('small');
            }
        }
    };

    const checkIfChipGroupSpansHidden = (hiddenSpanIndexes: number[], expectedTotalSpanCount: number) => {
        const hiddenClassName = 'hidden-chip';
        const spans = getChipGroupSpans();
        expect(spans.length).toBe(expectedTotalSpanCount);

        spans.forEach((span, index) => {
            if (hiddenSpanIndexes.includes(index)) {
                expect(span).toHaveClassName(hiddenClassName);
            } else {
                expect(span).not.toHaveClassName(hiddenClassName);
            }
            const chip = span.find(DotChip);
            expect(chip).toExist();
            expect(chip.props().size).toBe('small');
        });
    };

    const mockUseChipGroup = (widths: UseChipGroupWidths) => {
        jest.spyOn(hooks, 'useChipGroup').mockImplementation(() => widths);
    };

    it('should render without errors', () => {
        mount();
        expect(wrapper.exists()).toBe(true);
    });

    it('should render chips for each label', () => {
        mockUseChipGroup({ chipCounterWidth: 0, containerWidth: 250, chipWidths: [100, 25, 25, 25] });
        mount();
        checkIfChipGroupSpansHidden([], 4);
        assertExceedingWidthChipSpan(false);
    });

    it('should hide chips which cannot fit the available width', () => {
        mockUseChipGroup({ chipCounterWidth: 40, containerWidth: 200, chipWidths: [60, 50, 100, 100] });
        mount();
        checkIfChipGroupSpansHidden([2, 3], 4);
        assertExceedingWidthChipSpan(true, '+2');
    });

    it('should render Chip with charactersLimit prop', () => {
        const chipCharactersLimit = 15;
        mount({ ...defaultProps, chipCharactersLimit });
        const chips = getChipGroupSpans().find(DotChip);
        chips.forEach((chip) => expect(chip.props().charactersLimit).toBe(chipCharactersLimit));
    });

    it('should always display at least one category', () => {
        mockUseChipGroup({ chipCounterWidth: 40, containerWidth: 50, chipWidths: [60, 50, 100, 100] });
        mount();
        checkIfChipGroupSpansHidden([1, 2, 3], 4);
    });

    describe('edge case gap width', () => {
        it('should calculate in gap widths and hide last element', () => {
            // 40 + 50 + 50 + 50 + 19 + (4 * 8 gap) = 241 (larger than container width)
            mockUseChipGroup({ chipCounterWidth: 40, containerWidth: 240, chipWidths: [50, 50, 50, 19] });
            mount();
            checkIfChipGroupSpansHidden([3], 4);
            assertExceedingWidthChipSpan(true, '+1');
        });

        it('should calculate in gap widths and still show all elements when total width equals container width, case 1', () => {
            // 40 + 50 + 50 + 50 + 18 + (4 * 8 gap) = 240 (same as container width)
            mockUseChipGroup({ chipCounterWidth: 40, containerWidth: 240, chipWidths: [50, 50, 50, 18] });
            mount();
            checkIfChipGroupSpansHidden([], 4);
            assertExceedingWidthChipSpan(false);
        });

        it('should calculate in gap widths and still show all elements when total width equals container width, case 2', () => {
            // 40 + 18 + 50 + 50 + 50 + (4 * 8 gap) = 240 (same as container width)
            mockUseChipGroup({ chipCounterWidth: 40, containerWidth: 240, chipWidths: [18, 50, 50, 50] });
            mount();
            checkIfChipGroupSpansHidden([], 4);
            assertExceedingWidthChipSpan(false);
        });
    });
});
