import React from 'react';
import { mount, shallow } from 'enzyme/build';
import constant from 'lodash/constant';
import includes from 'lodash/includes';
import noop from 'lodash/noop';
import { XlrTags } from '../../../components';
import { DEFAULT_LABEL_FIELD, DEFAULT_PLACEHOLDER } from './xlr-tags-constants';
import { keyCodes } from '../../../constants/key-codes';

const defaults = {
    tags: [{ id: 'Apple', text: 'Apple', readOnly: false }],
    suggestions: [
        { id: 'Banana', text: 'Banana', readOnly: false },
        { id: 'Apple', text: 'Apple', readOnly: false },
        { id: 'Apricot', text: 'Apricot', readOnly: false },
        { id: 'Pear', text: 'Pear', readOnly: false },
        { id: 'Peach', text: 'Peach', readOnly: false },
    ],
};

const DOWN_ARROW_KEY_CODE = 40;
const ENTER_ARROW_KEY_CODE = 13;

function mockItem(overrides) {
    const props = Object.assign({}, defaults, overrides);
    return <XlrTags {...props} />;
}

describe('XlrTags component', () => {
    test('should render with expected props', () => {
        const xlrTagsComponent = shallow(mockItem());
        const expectedProps = {
            placeholder: DEFAULT_PLACEHOLDER,
            suggestions: [],
            delimiters: [keyCodes.ENTER],
            autofocus: true,
            labelField: DEFAULT_LABEL_FIELD,
            handleDelete: noop,
            handleAddition: noop,
            allowDeleteFromEmptyInput: true,
            allowAddFromPaste: true,
            allowAddOnlyFromSuggestion: false,
            resetInputOnDelete: true,
            autocomplete: false,
            readOnly: false,
            ...defaults,
            allowUnique: true,
        };
        expect(xlrTagsComponent.instance().props).toEqual(expectedProps);
    });

    test('should update the class when the prop classNames changes', () => {
        const xlrTagsComponent = mount(
            mockItem({
                classNames: {
                    tag: 'initial',
                },
            }),
        );
        expect(xlrTagsComponent.find('.initial').length).toBe(1);
        xlrTagsComponent.setProps({
            classNames: {
                tag: 'changed',
            },
        });
        expect(xlrTagsComponent.find('.changed').length).toBe(1);
    });

    test('focus on input by default', () => {
        const xlrTagsComponent = mount(mockItem(), { attachTo: document.body });
        expect(document.activeElement.tagName).toBe('INPUT');
        expect(document.activeElement.className).toBe('tag-input');
        xlrTagsComponent.unmount();
    });

    test('should not focus on input if autofocus is false', () => {
        const xlrTagsComponent = mount(mockItem({ autofocus: false }));
        expect(document.activeElement.tagName).toBe('BODY');
        xlrTagsComponent.unmount();
    });

    test('should not focus on input if readOnly is true', () => {
        const xlrTagsComponent = mount(mockItem({ autofocus: false }));
        expect(document.activeElement.tagName).toBe('BODY');
        xlrTagsComponent.unmount();
    });

    test('shows the classnames of children properly', () => {
        const xlrTagsComponent = mount(mockItem());
        expect(xlrTagsComponent.find('.tags').length).toBe(1);
        expect(xlrTagsComponent.find('.tag').length).toBe(1);
        expect(xlrTagsComponent.find('.tag-new').length).toBe(1);
        expect(xlrTagsComponent.find('.tag-input').length).toBe(1);
    });

    test('renders preselected tags properly', () => {
        const xlrTagsComponent = mount(mockItem());
        expect(xlrTagsComponent.text()).toContain('Apple');
    });

    test('invokes the onBlur event', () => {
        const handleInputBlur = jest.fn();
        const xlrTagsComponent = mount(mockItem());

        // Won't be invoked as there's no `handleInputBlur` event yet.
        xlrTagsComponent.find('.tag-input').simulate('blur');
        expect(handleInputBlur).toHaveBeenCalledTimes(0);

        // Will be invoked despite the input being empty.
        xlrTagsComponent.setProps({ handleInputBlur });
        xlrTagsComponent.find('.tag-input').simulate('blur');
        expect(handleInputBlur).toHaveBeenCalledTimes(1);
        expect(handleInputBlur).toHaveBeenCalledWith('');
        expect(xlrTagsComponent.find('.tag-input').get(0).value).toBeUndefined();
    });

    test('invokes the onFocus event', () => {
        const handleInputFocus = jest.fn();
        const xlrTagsComponent = mount(mockItem({ inputValue: 'Example' }));

        xlrTagsComponent.setProps({ handleInputFocus });
        xlrTagsComponent.find('.tag-input').simulate('focus');
        expect(handleInputFocus).toHaveBeenCalledTimes(1);
        expect(handleInputFocus).toHaveBeenCalledWith('Example');
    });

    test('invokes the onBlur event when input has value', () => {
        const handleInputBlur = jest.fn();
        const xlrTagsComponent = mount(mockItem({ inputValue: 'Example' }));

        // Will also be invoked for when the input has a value.
        xlrTagsComponent.setProps({ handleInputBlur });
        xlrTagsComponent.find('.tag-input').simulate('blur');
        expect(handleInputBlur).toHaveBeenCalledTimes(1);
        expect(handleInputBlur).toHaveBeenCalledWith('Example');
        expect(xlrTagsComponent.find('.tag-input').get(0).value).toBeUndefined();
    });

    describe('tests handlePaste', () => {
        test('should not add new tag when allowAddFromPaste is false', () => {
            const actual = [];
            const xlrTagsComponent = mount(
                mockItem({
                    allowAddFromPaste: false,
                    handleAddition(tag) {
                        actual.push(tag);
                    },
                }),
            );

            const $input = xlrTagsComponent.find('.tag-input');

            $input.simulate('paste', {
                clipboardData: {
                    getData: constant('Banana'),
                },
            });

            expect(actual.length).toBe(0);
            expect(actual).toEqual(expect.not.arrayContaining(['Banana']));
        });

        test('should split the clipboard on delimiters', () => {
            const Keys = {
                TAB: 9,
                SPACE: 32,
                COMMA: 188,
            };

            const tags = [];
            const xlrTagsComponent = mount(
                mockItem({
                    delimiters: [Keys.TAB, Keys.SPACE, Keys.COMMA],
                    handleAddition(tag) {
                        tags.push(tag);
                    },
                    tags,
                }),
            );

            const $input = xlrTagsComponent.find('.tag-input');

            $input.simulate('paste', {
                clipboardData: {
                    getData: constant('Banana,Apple,Apricot\nOrange Blueberry,Pear,Peach\tKiwi'),
                },
            });

            const expected = ['Banana', 'Apple', 'Apricot\nOrange', 'Blueberry', 'Pear', 'Peach', 'Kiwi'].map((value) => ({
                id: value,
                text: value,
                readOnly: false,
            }));

            expect(tags).toEqual(expected);
        });

        test('should not allow duplicate tags', () => {
            const Keys = {
                TAB: 9,
                SPACE: 32,
                COMMA: 188,
            };

            const tags = [...defaults.tags];
            const xlrTagsComponent = mount(
                mockItem({
                    delimiters: [Keys.TAB, Keys.SPACE, Keys.COMMA],
                    handleAddition(tag) {
                        tags.push(tag);
                    },
                    tags,
                }),
            );

            const $input = xlrTagsComponent.find('.tag-input');

            $input.simulate('paste', {
                clipboardData: {
                    getData: constant('Banana,Apple,Banana'),
                },
            });

            // Note that 'Apple' and 'Banana' are only included once in the expected list
            const expected = ['Apple', 'Banana'].map((value) => ({
                id: value,
                text: value,
                readOnly: false,
            }));

            expect(tags).toEqual(expected);
        });

        test('should allow pasting text only up to maxLength characters', () => {
            const tags = [];
            const maxLength = 5;
            const inputValue = 'Thimbleberry';

            const xlrTagsComponent = mount(
                mockItem({
                    handleAddition(tag) {
                        tags.push(tag);
                    },
                    maxLength,
                }),
            );

            const $input = xlrTagsComponent.find('.tag-input');

            $input.simulate('paste', {
                clipboardData: {
                    getData: () => inputValue,
                },
            });

            const clipboardText = inputValue.substr(0, maxLength);
            const expected = [clipboardText].map((value) => ({
                id: value,
                text: value,
                readOnly: false,
            }));
            expect(tags).toEqual(expected);
        });
    });

    test('should not allow duplicate tags', () => {
        const actual = [];
        const xlrTagsComponent = mount(
            mockItem({
                handleAddition(tag) {
                    actual.push(tag);
                },
            }),
        );

        expect(xlrTagsComponent.instance().props.tags).toEqual(defaults.tags);
        const $input = xlrTagsComponent.find('.tag-input');
        $input.simulate('change', { target: { value: 'Apple' } });
        $input.simulate('keyDown', { keyCode: ENTER_ARROW_KEY_CODE });
        expect(actual.length).toBe(0);
    });

    test('should not add empty tag when down arrow is clicked followed by enter key', () => {
        const actual = [];
        const xlrTagsComponent = mount(
            mockItem({
                handleAddition(tag) {
                    actual.push(tag);
                },
                suggestions: [],
            }),
        );

        expect(xlrTagsComponent.instance().props.tags).toEqual(defaults.tags);

        const $input = xlrTagsComponent.find('.tag-input');
        $input.simulate('keyDown', { keyCode: DOWN_ARROW_KEY_CODE });
        $input.simulate('keyDown', { keyCode: ENTER_ARROW_KEY_CODE });
        expect(actual.length).toBe(0);
    });

    // this test will fail if console.error occurs
    test('should not set any property of this.textInput when readOnly', () => {
        // eslint-disable-next-line no-console, angular/log
        console.error = jest.fn((error) => {
            throw error;
        });

        const xlrTagsComponent = mount(mockItem({ readOnly: true, resetInputOnDelete: false }));
        const $tag = xlrTagsComponent.find('.tag-label');
        $tag.simulate('click');
    });

    test('should fail the test if two tags have same key', () => {
        // eslint-disable-next-line no-console, angular/log
        console.warn = jest.fn(() => {
            throw 'Error';
        });

        let modifiedTags = [...defaults.tags, { id: 'NewYork', text: 'NewYork', readOnly: false }, { id: 'Austria', text: 'Austria', readOnly: false }];
        const xlrTagsComponent = mount(
            mockItem({
                tags: modifiedTags,
                handleDelete: (i) => {
                    modifiedTags = modifiedTags.filter((tag, index) => index !== i);
                },
            }),
        );
        //remove Apple
        xlrTagsComponent.find('.tag-close').at(0).simulate('click');
        //remove NewYork
        xlrTagsComponent.find('.tag-close').at(1).simulate('click');
        xlrTagsComponent.setProps({ tags: modifiedTags });
        const $input = xlrTagsComponent.find('.tag-input');
        $input.simulate('change', { target: { value: 'Hello' } });
        $input.simulate('keyDown', { keyCode: DOWN_ARROW_KEY_CODE });
    });

    describe('render tags correctly when html passed in  text attribute', () => {
        let modifiedTags = [];
        let handleAddition;
        let actual;

        beforeEach(() => {
            actual = [];
            modifiedTags = [
                ...defaults.tags,
                { id: '1', text: <span style={{ color: 'red' }}> NewYork</span>, readOnly: false },
                { id: '2', text: <span style={{ color: 'blue' }}> Austria</span>, readOnly: false },
            ];
            handleAddition = ({ id, text }) => {
                actual.push({
                    id,
                    text: <span style={{ color: 'yellow' }}>{text}</span>,
                    readOnly: false,
                });
            };
        });

        test('should render tags correctly', () => {
            const xlrTagsComponent = mount(
                mockItem({
                    tags: modifiedTags,
                }),
            );
            expect(xlrTagsComponent.instance().props.tags).toEqual(modifiedTags);
        });

        test('allow adding tag which is not in the list', () => {
            const xlrTagsComponent = mount(
                mockItem({
                    tags: modifiedTags,
                    handleAddition,
                }),
            );
            const $input = xlrTagsComponent.find('.tag-input');
            $input.simulate('change', { target: { value: 'Custom tag' } });

            $input.simulate('keyDown', { keyCode: ENTER_ARROW_KEY_CODE });
            expect(actual.length).toBe(1);
            expect(React.isValidElement(actual[0].text)).toBe(true);
        });
    });

    describe('autocomplete/suggestions filtering', () => {
        test('updates suggestions state if the suggestions prop changes', () => {
            const xlrTagsComponent = mount(mockItem());
            const $input = xlrTagsComponent.find('.tag-input');

            $input.simulate('change', { target: { value: 'apr' } });
            expect(xlrTagsComponent.state().suggestions).toEqual([{ id: 'Apricot', text: 'Apricot', readOnly: false }]);

            xlrTagsComponent.setProps({
                suggestions: [
                    { id: 'Papaya', text: 'Papaya' },
                    { id: 'Paprika', text: 'Paprika' },
                ],
            });
            expect(xlrTagsComponent.state().suggestions).toEqual([{ id: 'Paprika', text: 'Paprika' }]);
        });

        test('updates suggestions state as expected based on default filter logic', () => {
            const xlrTagsComponent = mount(mockItem());
            const $input = xlrTagsComponent.find('.tag-input');

            expect(xlrTagsComponent.state().suggestions).toEqual(defaults.suggestions);

            $input.simulate('change', { target: { value: 'or' } });
            expect(xlrTagsComponent.state().suggestions).toEqual([]);

            $input.simulate('change', { target: { value: 'ea' } });
            expect(xlrTagsComponent.state().suggestions).toEqual([
                { id: 'Pear', text: 'Pear', readOnly: false },
                { id: 'Peach', text: 'Peach', readOnly: false },
            ]);

            $input.simulate('change', { target: { value: 'apr' } });
            expect(xlrTagsComponent.state().suggestions).toEqual([{ id: 'Apricot', text: 'Apricot', readOnly: false }]);
        });

        test('updates suggestions state as expected based on custom filter logic', () => {
            const xlrTagsComponent = mount(
                mockItem({
                    handleFilterSuggestions: (query, suggestions) => {
                        return suggestions.filter((suggestion) => {
                            return includes(suggestion.text.toLowerCase(), query.toLowerCase());
                        });
                    },
                }),
            );
            const $input = xlrTagsComponent.find('.tag-input');

            expect(xlrTagsComponent.state().suggestions).toEqual(defaults.suggestions);

            $input.simulate('change', { target: { value: 'Ea' } });
            expect(xlrTagsComponent.state().suggestions).toEqual([
                { id: 'Pear', text: 'Pear', readOnly: false },
                { id: 'Peach', text: 'Peach', readOnly: false },
            ]);

            $input.simulate('change', { target: { value: 'apr' } });
            expect(xlrTagsComponent.state().suggestions).toEqual([{ id: 'Apricot', text: 'Apricot', readOnly: false }]);
        });

        test('updates selectedIndex state as expected based on changing suggestions', () => {
            const xlrTagsComponent = mount(
                mockItem({
                    autocomplete: true,
                    handleFilterSuggestions: (query, suggestions) => {
                        return suggestions.filter((suggestion) => {
                            return includes(suggestion.text.toLowerCase(), query.toLowerCase());
                        });
                    },
                }),
            );
            const $input = xlrTagsComponent.find('.tag-input');

            expect(xlrTagsComponent.state().suggestions).toEqual(defaults.suggestions);

            $input.simulate('change', { target: { value: 'Ea' } });
            $input.simulate('focus');
            $input.simulate('keyDown', { keyCode: DOWN_ARROW_KEY_CODE });
            $input.simulate('keyDown', { keyCode: DOWN_ARROW_KEY_CODE });
            expect(xlrTagsComponent.state().suggestions).toEqual([
                { id: 'Pear', text: 'Pear', readOnly: false },
                { id: 'Peach', text: 'Peach', readOnly: false },
            ]);
            expect(xlrTagsComponent.state().selectedIndex).toBe(1);
            $input.simulate('change', { target: { value: 'Each' } });
            expect(xlrTagsComponent.state().suggestions).toEqual([{ id: 'Peach', text: 'Peach', readOnly: false }]);
            expect(xlrTagsComponent.state().selectedIndex).toBe(0);
        });

        test('selects the correct suggestion using the keyboard when minQueryLength is set to 0', () => {
            const actual = [];
            const xlrTagsComponent = mount(
                mockItem({
                    query: '',
                    minQueryLength: 0,
                    handleAddition(tag) {
                        actual.push(tag);
                    },
                }),
            );
            const $input = xlrTagsComponent.find('.tag-input');

            $input.simulate('keyDown', { keyCode: DOWN_ARROW_KEY_CODE });
            $input.simulate('keyDown', { keyCode: DOWN_ARROW_KEY_CODE });
            $input.simulate('keyDown', { keyCode: DOWN_ARROW_KEY_CODE });
            $input.simulate('keyDown', { keyCode: ENTER_ARROW_KEY_CODE });
            expect(actual).toEqual([{ id: 'Apricot', text: 'Apricot', readOnly: false }]);

            xlrTagsComponent.unmount();
        });

        test('handles addition when using default suggestions filter', () => {
            const actual = [];
            const xlrTagsComponent = mount(
                mockItem({
                    autocomplete: true,
                    handleAddition(tag) {
                        actual.push(tag);
                    },
                }),
            );
            const $input = xlrTagsComponent.find('.tag-input');

            $input.simulate('change', { target: { value: 'Or' } });
            $input.simulate('focus');
            $input.simulate('keyDown', { keyCode: ENTER_ARROW_KEY_CODE });
            expect(actual).toEqual([{ id: 'Or', text: 'Or', readOnly: false }]);
        });

        test('should add tag with custom label field and default suggestion filter', () => {
            const labelField = 'name';
            const mapper = (data) => ({ id: data.id, name: data.text, readOnly: data.readOnly });
            const tags = defaults.tags.map(mapper);
            const suggestions = defaults.suggestions.map(mapper);

            const actual = [];
            const xlrTagsComponent = mount(
                mockItem({
                    autocomplete: true,
                    handleAddition(tag) {
                        actual.push(tag);
                    },
                    labelField,
                    tags,
                    suggestions,
                }),
            );
            const $input = xlrTagsComponent.find('.tag-input');
            $input.simulate('change', { target: { value: 'Or' } });
            $input.simulate('focus');
            $input.simulate('keyDown', { keyCode: ENTER_ARROW_KEY_CODE });
            expect(actual).toEqual([{ id: 'Or', name: 'Or', readOnly: false }]);
        });

        test('should select the correct suggestion using the keyboard when label is custom', () => {
            const labelField = 'name';
            const mapper = (data) => ({ id: data.id, name: data.text, readOnly: data.readOnly });
            const actual = [];
            const suggestions = defaults.suggestions.map(mapper);

            const xlrTagsComponent = mount(
                mockItem({
                    labelField,
                    tags: actual,
                    suggestions,
                    handleAddition(tag) {
                        actual.push(tag);
                    },
                }),
            );

            const $input = xlrTagsComponent.find('.tag-input');

            $input.simulate('change', { target: { value: 'Ap' } });
            $input.simulate('keyDown', { keyCode: DOWN_ARROW_KEY_CODE });
            $input.simulate('keyDown', { keyCode: DOWN_ARROW_KEY_CODE });
            $input.simulate('keyDown', { keyCode: ENTER_ARROW_KEY_CODE });
            expect(actual).toEqual([{ id: 'Apricot', [labelField]: 'Apricot', readOnly: false }]);
            xlrTagsComponent.unmount();
        });
    });

    test('should render default tags with custom label field', () => {
        const labelField = 'name';
        const mapper = (data) => ({ id: data.id, name: data.text, readOnly: data.readOnly });
        const tags = defaults.tags.map(mapper);
        const suggestions = defaults.suggestions.map(mapper);

        const expectedText = tags[0][labelField];

        const props = {
            labelField,
            tags,
            suggestions,
        };

        const xlrTagsComponent = mount(mockItem(props));
        expect(xlrTagsComponent.text()).toEqual(expectedText);
        xlrTagsComponent.unmount();
    });

    test('should allow duplicate tags when allowUnique is false', () => {
        const actual = [];
        const xlrTagsComponent = mount(
            mockItem({
                handleAddition(tag) {
                    actual.push(tag);
                },
                allowUnique: false,
            }),
        );

        expect(xlrTagsComponent.instance().props.tags).toEqual(defaults.tags);
        const $input = xlrTagsComponent.find('.tag-input');
        $input.simulate('change', { target: { value: 'Apple' } });
        $input.simulate('keyDown', { keyCode: ENTER_ARROW_KEY_CODE });
        expect(actual).toEqual([
            {
                id: 'Apple',
                text: 'Apple',
                readOnly: false,
            },
        ]);
    });

    test('should not allow arbitrary tags when allowAddOnlyFromSuggestion is true', () => {
        const handleAddition = jest.fn();
        const xlrTagsComponent = mount(
            mockItem({
                handleAddition,
                allowAddOnlyFromSuggestion: true,
            }),
        );

        expect(xlrTagsComponent.instance().props.tags).toEqual(defaults.tags);
        const $input = xlrTagsComponent.find('.tag-input');
        $input.simulate('change', { target: { value: 'another' } });
        $input.simulate('keyDown', { keyCode: ENTER_ARROW_KEY_CODE });
        expect(handleAddition).toHaveBeenCalledTimes(0);
    });
});
