import React, { MouseEvent, useEffect, useState } from 'react';
import isNil from 'lodash/isNil';
import remarkGfm from 'remark-gfm';
import remarkBreaks from 'remark-breaks';
import rehypeSanitize from 'rehype-sanitize';
import rehypeStringify from 'rehype-stringify';
import remarkParse from 'remark-parse';
import remarkRehype from 'remark-rehype';
import { unified } from 'unified';
import classNames from 'classnames';
import { DotButton, DotIconButton, DotInputText, DotTooltip, DotTypography } from '@digital-ai/dot-components';
import { HelpButton } from '../../../features/main-navigation/help-button/help-button';
import { OptionSelectorInput } from '../../../features/common/components/options-selector/option-selector-input.component';
import { ReleaseVariable, User, Variable } from '../../../types';
import { VariablesInterpolatorFactory } from '../../../features/tasks/types/angular';
import getAngularService from '../../../features/common/services/angular-accessor';
import { interpolateLinks, interpolateUsers, wrapMentionsText } from './helper';
import './textarea-markdown.component.less';

export const MAX_LINES = 300;

export interface TextareaMarkdownProps {
    className?: string;
    enableMentions?: boolean;
    enableVariables?: boolean;
    inputId: string;
    isReadOnly?: boolean;
    isValueRequired?: boolean;
    maxRows?: number;
    name: string;
    onDelete?: (event: MouseEvent<HTMLButtonElement>) => void;
    onTextChange?: (text: string) => void;
    placeholder?: string;
    releaseVariables?: ReleaseVariable;
    rows: number;
    text: string;
    users?: User[];
    variables?: Variable[];
}

export const TextAreaMarkdown = ({
    className,
    enableMentions = false,
    enableVariables = false,
    inputId,
    isReadOnly = false,
    isValueRequired = true,
    maxRows,
    name,
    onDelete,
    onTextChange,
    placeholder,
    rows,
    text,
    users = [],
    variables,
    releaseVariables,
}: TextareaMarkdownProps) => {
    const [editing, setEditing] = useState<boolean>(false);
    const [originalText, setOriginalText] = useState<string>(text);
    const [currentText, setCurrentText] = useState<string>(text);
    // states for html
    const [parsedHtml, setParsedHtml] = useState<string>('');
    const [markdownText, setMarkdownText] = useState<string>('');

    const isSaveButtonDisabled = isValueRequired && (isNil(currentText) || !currentText.trim());
    const variablesInterpolator = getAngularService('VariablesInterpolator') as never as VariablesInterpolatorFactory;

    useEffect(() => {
        if (!markdownText) return;
        const numberOfLines = (markdownText.match(/\n/g) || '').length + 1;
        const markdownParser = unified().use(remarkParse).use(remarkGfm);
        if (numberOfLines < MAX_LINES) {
            markdownParser.use(remarkBreaks);
        }
        markdownParser
            .use(remarkRehype)
            .use(rehypeSanitize)
            .use(rehypeStringify)
            .process(markdownText)
            .then((html) => {
                let htmlReplaced = html.value as string;
                if (enableVariables) {
                    // interpolateInHtml function is always there but i have to fix a bunch of tests, for later...
                    htmlReplaced = variablesInterpolator.interpolateInHtml?.(releaseVariables ?? {}, html.value as string) ?? htmlReplaced;
                }
                if (enableMentions) {
                    htmlReplaced = interpolateUsers(htmlReplaced, users);
                }
                htmlReplaced = interpolateLinks(htmlReplaced);
                setParsedHtml(htmlReplaced);
            });
    }, [markdownText]);

    useEffect(() => {
        if (!editing) {
            const content = enableMentions ? wrapMentionsText(currentText) : currentText;
            setMarkdownText(content);
        }
    }, [editing, enableMentions, currentText]);

    useEffect(() => {
        setOriginalText(text);
        setCurrentText(text);
    }, [text]);

    useEffect(() => {
        if (isReadOnly) {
            revertChanges();
        }
    }, [isReadOnly]);

    const saveChanges = () => {
        const trimmedText = currentText.trim();
        onTextChange && onTextChange(trimmedText);
        setCurrentText(trimmedText);
        setOriginalText(trimmedText);
        setEditing(false);
    };

    const revertChanges = () => {
        setCurrentText(originalText);
        setEditing(false);
    };

    const handleDelete = (event: MouseEvent<HTMLButtonElement>) => {
        onDelete && onDelete(event);
        setEditing(false);
    };

    const renderTextEditor = () => (
        <div>
            {enableMentions || enableVariables ? (
                <OptionSelectorInput
                    enableMentions={enableMentions}
                    enableVariables={enableVariables}
                    id={inputId}
                    maxRows={maxRows}
                    multiline={true}
                    name={name}
                    onChange={(inputValue) => setCurrentText(inputValue)}
                    rows={rows}
                    users={users}
                    value={currentText}
                    variables={variables}
                />
            ) : (
                <DotInputText
                    autoFocus
                    id={inputId}
                    maxRows={maxRows}
                    minRows={rows}
                    multiline={true}
                    name={name}
                    onChange={(e) => setCurrentText(e.target.value)}
                    value={currentText}
                />
            )}
            <div className="actions-wrapper">
                <DotButton data-testid="cancel-button" onClick={revertChanges} type="text">
                    Cancel
                </DotButton>
                <DotButton data-testid="save-button" disabled={isSaveButtonDisabled} onClick={saveChanges} type="primary">
                    Save
                </DotButton>
                <HelpButton helpLink="how-to/using-markdown-in-xl-release.html" />
            </div>
        </div>
    );

    const renderMarkDownPreview = () => (
        <div
            className={classNames('markdown-viewer', { disabled: isReadOnly })}
            onDoubleClick={() => !isReadOnly && setEditing(true)}
            role="button"
            tabIndex={isReadOnly ? undefined : 0}
        >
            {!isReadOnly ? (
                <div className="markdown-viewer-actions">
                    {onDelete && originalText && <DotIconButton data-testid="delete-button" iconId="delete" onClick={handleDelete} size="small" />}
                    <DotIconButton data-testid="edit-button" iconId="edit" onClick={() => setEditing(true)} size="small" />
                </div>
            ) : null}
            {currentText ? (
                <DotTooltip disablePortal={true} title={!isReadOnly ? 'Double-click to edit' : undefined}>
                    <div className="markdown-wrapper">
                        <div dangerouslySetInnerHTML={{ __html: parsedHtml }} />
                    </div>
                </DotTooltip>
            ) : (
                !isReadOnly && (
                    <DotTooltip disablePortal={true} title="Double-click to edit">
                        <div className="placeholder-wrapper" onDoubleClick={() => setEditing(true)}>
                            <DotTypography className="placeholder" variant="body1">
                                {placeholder}
                            </DotTypography>
                        </div>
                    </DotTooltip>
                )
            )}
        </div>
    );

    return <div className={classNames('markdown-switcher', className)}>{editing ? renderTextEditor() : renderMarkDownPreview()}</div>;
};
