import React, { useState } from 'react';
import { Provider, useDispatch } from 'react-redux';
import { DotButton, DotEmptyFilterState, DotEmptyState, DotThemeProvider } from '@digital-ai/dot-components';
import { store } from '@xlr-ui/app/js/app-config';
import getAngularService from '@xlr-ui/app/features/common/services/angular-accessor';
import { UploadService } from '@xlr-ui/app/features/tasks/types/angular';
import { navigation } from '@xlr-ui/app/features/main-navigation/ducks/navigation.reducer';
import { ReactImportTemplateModalWrapper } from '@xlr-ui/app/features/templates/import-template-modal-wrapper.component';
import './template-list-empty-state.component.less';

const { navigateTo } = navigation.actions;

export interface TemplateListEmptyStateProps {
    filtered?: boolean;
    folder?: { id: string };
    hasCreateTemplatePermission?: boolean;
    onImportTemplateSuccess?: () => void;
}

export const TemplateListEmptyStateContent = ({
    filtered = false,
    folder,
    hasCreateTemplatePermission,
    onImportTemplateSuccess,
}: TemplateListEmptyStateProps) => {
    const uploadService = getAngularService('UploadService') as never as UploadService;
    const [showImportTemplateModal, setShowImportTemplateModal] = useState(false);
    const dispatch = useDispatch();

    const onNewTemplate = () => {
        if (folder) {
            dispatch(navigateTo({ pathSuffix: `folders/${folder.id}/templates?action=create` }));
        } else {
            dispatch(navigateTo({ pathSuffix: 'templates/create' }));
        }
    };

    const onImportTemplate = () => {
        uploadService.reset();
        setShowImportTemplateModal(true);
    };

    const modalProps = {
        folder,
        onHide: () => setShowImportTemplateModal(false),
        onImportTemplateSuccess,
        show: showImportTemplateModal,
    };

    return (
        <DotThemeProvider>
            {filtered ? (
                <DotEmptyFilterState />
            ) : (
                <>
                    <DotEmptyState
                        illustrationId="add-new-grid"
                        subtitle="Design a release process template by creating a new one or importing an existing template."
                        title="Create release template"
                    />
                    {hasCreateTemplatePermission && (
                        <div className="empty-state-buttons">
                            <DotButton onClick={onImportTemplate} type="text">
                                Import template
                            </DotButton>
                            <DotButton onClick={onNewTemplate}>New template</DotButton>
                            <ReactImportTemplateModalWrapper {...modalProps} />
                        </div>
                    )}
                </>
            )}
        </DotThemeProvider>
    );
};

export const TemplateListEmptyState = (props: TemplateListEmptyStateProps) => (
    <Provider store={store}>
        <TemplateListEmptyStateContent {...props} />
    </Provider>
);
