import React from 'react';
import { DotThemeProvider, DotTable, DotLink } from '@digital-ai/dot-components';
import PropTypes from 'prop-types';
import IdsFactory from '../../../js/util/ids';

export const ConfigurationDeleteReferenceComponent = ({ data, responseMessage }) => {
    const getCiReferenceLink = (reference) => {
        switch (true) {
            case reference.referenceType.includes('Release'): {
                const releaseIdFromReleaseReference = IdsFactory().releaseIdFrom(reference.id);
                const releaseInternalId = IdsFactory().toInternalId(releaseIdFromReleaseReference);
                return `#/releases/${releaseInternalId}`;
            }
            case reference.referenceType.includes('Task'): {
                const releaseId = IdsFactory().releaseIdFrom(reference.id);
                const phaseTaskId = IdsFactory().phaseTaskIdFrom(reference.id);
                const internalId = IdsFactory().toInternalId(phaseTaskId);
                return `#/releases/${releaseId}?openTaskDetailsModal=${internalId}`;
            }
            case reference.referenceType.includes('Template'): {
                const formTemplate = reference.id.split('/');
                return `#/templates/${formTemplate[1]}-${formTemplate[2]}`;
            }
            case reference.referenceType.includes('Trigger'): {
                const formTrigger = reference.id.split('/');
                return `#/trigger-management/${formTrigger[2]}?folderId=${formTrigger[0]}/${formTrigger[1]}`;
            }
            case reference.referenceType.includes('Configuration'): {
                if (reference.folderId === null) return '#/configuration';
                return `#/folders/${reference.folderId}/configuration`;
            }
            default:
                return '';
        }
    };

    const getConfigurationReferenceData = () => {
        const tableData = [];
        data.forEach((reference) => {
            const link = getCiReferenceLink(reference);
            const warningData = {
                id: reference.id,
                rowData: {
                    title: reference.title,
                    link: reference.referenceType.includes('Configuration') ? (
                        <DotLink href={link} target="_self">
                            {reference.folderTitle} connections
                        </DotLink>
                    ) : (
                        <DotLink href={link} target="_self">
                            Click Here
                        </DotLink>
                    ),
                },
            };
            tableData.push(warningData);
        });
        return tableData;
    };

    const columns = () => {
        return [
            {
                id: 'title',
                label: 'Title',
                width: '50%',
            },
            {
                id: 'link',
                label: 'Referenced at',
                width: '50%',
            },
        ];
    };

    return data.length !== 0 ? (
        <DotThemeProvider>
            <DotTable columns={columns()} data={getConfigurationReferenceData()} maxHeight="100%" order="asc" orderBy="title" rowsPerPage={10} />
        </DotThemeProvider>
    ) : (
        <div>
            <p>{responseMessage}</p>
        </div>
    );
};

ConfigurationDeleteReferenceComponent.propTypes = {
    data: PropTypes.arrayOf(
        PropTypes.shape({
            id: PropTypes.string,
            title: PropTypes.string,
        }),
    ),
    responseMessage: PropTypes.string,
};
